package ru.yandex.direct.jobs.freelancers.moderation.receiving;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.freelancers.moderation.receiving.model.BaseModerationVerdict;
import ru.yandex.direct.jobs.freelancers.moderation.receiving.model.VerdictType;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.utils.json.JSonSplitter.splitJsons;

/**
 * Класс реализующий получение из лог-брокера вердиктов модерации на карточки фрилансеров.
 */
class VerdictReceiver {
    private static final Logger logger = LoggerFactory.getLogger(VerdictReceiver.class);

    private static final String DIRECT_VERDICT_SERVICE = "direct";

    private final SyncConsumer syncConsumer;
    private final Map<VerdictType, VerdictHandler> verdictHandlersByVerdictType;

    private int lbMessageCount = 0;
    private int receivedVerdictCount = 0;
    private int appliedVerdictCount = 0;

    VerdictReceiver(SyncConsumer syncConsumer,
                    Map<VerdictType, VerdictHandler> verdictHandlersByVerdictType) {
        this.syncConsumer = syncConsumer;
        this.verdictHandlersByVerdictType = verdictHandlersByVerdictType;
    }

    int getLbMessageCount() {
        return lbMessageCount;
    }

    int getReceivedVerdictCount() {
        return receivedVerdictCount;
    }

    int getAppliedVerdictCount() {
        return appliedVerdictCount;
    }

    void readAndApply() throws TimeoutException {
        for (ConsumerReadResponse response = readResponse(); response != null; response = readResponse()) {
            lbMessageCount++;
            Long lastCookie = response.getCookie();
            List<MessageBatch> batches = response.getBatches();
            for (MessageBatch batch : batches) {
                List<MessageData> batchMessageData = batch.getMessageData();
                for (MessageData messageData : batchMessageData) {
                    byte[] bytes = messageData.getDecompressedData();
                    String strData = new String(bytes, StandardCharsets.UTF_8);
                    String[] jsons = splitJsons(strData);
                    for (String jsonMessage : jsons) {
                        processMessage(jsonMessage);
                    }
                }
            }
            commit(lastCookie);
        }
    }

    private void commit(Long lastCookie) throws TimeoutException {
        try {
            syncConsumer.commit(lastCookie);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private void processMessage(String jsonMessage) {
        if (isEmpty(jsonMessage)) {
            return;
        }
        VerdictHandler verdictHandler = getVerdictHandler(jsonMessage);
        if (verdictHandler == null) {
            return;
        }
        receivedVerdictCount++;
        try {
            boolean result = verdictHandler.applyVerdict(jsonMessage);
            if (result) {
                appliedVerdictCount++;
            }
        } catch (IllegalArgumentException ex) {
            logger.error("LogBroker message applying was ended with exception. Message: {}", jsonMessage, ex);
        }
    }

    private VerdictHandler getVerdictHandler(String jsonMessage) {
        BaseModerationVerdict baseVerdict;
        try {
            baseVerdict = JsonUtils.fromJson(jsonMessage, BaseModerationVerdict.class);
        } catch (IllegalArgumentException e) {
            logger.error("LogBroker message cannot be converted to BaseModerationVerdict. Message: {}", jsonMessage, e);
            return null;
        }
        String verdictService = baseVerdict.getService();
        if (!verdictService.equals(DIRECT_VERDICT_SERVICE)) {
            logger.warn("Got message for unknown \"service\": \"{}\". Message: {}", verdictService, jsonMessage);
            return null;
        }
        String strVerdictType = baseVerdict.getType();
        VerdictType verdictType;
        try {
            verdictType = Enum.valueOf(VerdictType.class, strVerdictType.toUpperCase());
        } catch (IllegalArgumentException | NullPointerException ex) {
            logger.error("LogBroker message contains unknown verdictType. Message: {}", jsonMessage, ex);
            return null;
        }

        VerdictHandler verdictHandler = verdictHandlersByVerdictType.get(verdictType);
        if (verdictHandler == null) {
            logger.error("Appropriate verdict handler has not been found for verdict type {}", verdictType);
        }
        return verdictHandler;
    }


    private ConsumerReadResponse readResponse() {
        try {
            return syncConsumer.read();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }
}
