package ru.yandex.direct.jobs.freelancers.moderation.sending;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.ClientAvatar;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate;
import ru.yandex.direct.core.entity.freelancer.repository.ClientAvatarRepository;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerCardRepository;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerClientAvatarService;
import ru.yandex.direct.jobs.freelancers.moderation.sending.model.CardModerationRequest;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.util.Collections.singletonList;

@Service
public class FreelancerCardModerationService {

    private final FreelancerCardRepository freelancerCardRepository;
    private final ClientAvatarRepository clientAvatarRepository;
    private final FreelancerClientAvatarService freelancerClientAvatarService;

    @Autowired
    public FreelancerCardModerationService(
            FreelancerCardRepository freelancerCardRepository,
            ClientAvatarRepository clientAvatarRepository,
            FreelancerClientAvatarService freelancerClientAvatarService) {
        this.freelancerCardRepository = freelancerCardRepository;
        this.clientAvatarRepository = clientAvatarRepository;
        this.freelancerClientAvatarService = freelancerClientAvatarService;
    }

    List<CardModerationRequest> getRequests(int shard) {
        List<FreelancerCard> draftFreelancerCards = freelancerCardRepository.getDraftFreelancerCards(shard);
        List<Long> avatarIds = StreamEx.of(draftFreelancerCards)
                .map(FreelancerCard::getAvatarId)
                .filter(Objects::nonNull)
                .toList();
        List<ClientAvatar> clientAvatars = clientAvatarRepository.get(shard, avatarIds);
        Map<Long, String> avatarUrlByAvatarId = StreamEx.of(clientAvatars)
                .toMap(ClientAvatar::getId, freelancerClientAvatarService::getAvatarUrl);
        return StreamEx.of(draftFreelancerCards)
                .map(card -> Converter.createRequest(card, avatarUrlByAvatarId))
                .toList();
    }

    /**
     * Ставит заданной карточке фрилансера статус модерации в IN_PROGRESS.
     * Обновляет статус только при соблюдении следующих условий:
     * - такая карта существует;
     * - в данный момент у неё статус модерации DRAFT;
     * - карта не в архивном состоянии.
     * При несоблюдении любого из условий метод завершается не изменяя состояние карточки.
     */
    void markCardAsInProgress(int shard, Long freelancerCardId) {
        List<FreelancerCard> freelancerCards =
                freelancerCardRepository.getFreelancerCards(shard, singletonList(freelancerCardId));
        FreelancerCard freelancerCard = Iterables.getFirst(freelancerCards, null);
        if (freelancerCard == null
                || !freelancerCard.getStatusModerate().equals(FreelancersCardStatusModerate.DRAFT)
                || freelancerCard.getIsArchived()) {
            return;
        }
        ModelChanges<FreelancerCard> changes = new ModelChanges<>(freelancerCardId, FreelancerCard.class);
        changes.process(FreelancersCardStatusModerate.IN_PROGRESS, FreelancerCard.STATUS_MODERATE);
        AppliedChanges<FreelancerCard> freelancerCardAppliedChanges = changes.applyTo(freelancerCard);
        freelancerCardRepository.updateFreelancerCards(shard, singletonList(freelancerCardAppliedChanges));
    }
}
