package ru.yandex.direct.jobs.freelancers.moderation.sending;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.jobs.freelancers.moderation.LogbrokerClientFactoryService;
import ru.yandex.direct.jobs.freelancers.moderation.LogbrokerProducerProperties;
import ru.yandex.direct.jobs.freelancers.moderation.sending.model.CardModerationRequest;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.FREELANCER_CARDS_MODERATION_SENDING_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;


/**
 * Джоба отправляющая карточку фрилансера на модерацию.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 25),
        needCheck = NonDevelopmentEnvironment.class,
        tags = {DIRECT_PRIORITY_1, JOBS_RELEASE_REGRESSION},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(periodInSeconds = 600, needSchedule = NonDevelopmentEnvironment.class)
@ParametersAreNonnullByDefault
public class FreelancerCardSendingJob extends DirectShardedJob {
    public static final String CONFIG_SECTION_NAME = "fl_card_lb_moderation_sending";
    private static final Logger logger = LoggerFactory.getLogger(FreelancerCardSendingJob.class);
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final LogbrokerProducerProperties producerProperties;
    private final FreelancerCardModerationService freelancerCardModerationService;
    private final LogbrokerClientFactoryService logbrokerClientFactoryService;
    private final LbSeqIdService lbSeqIdService;

    @Autowired
    public FreelancerCardSendingJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            DirectConfig directConfig,
            FreelancerCardModerationService freelancerCardModerationService,
            LogbrokerClientFactoryService logbrokerClientFactoryService,
            LbSeqIdService lbSeqIdService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.producerProperties = LogbrokerProducerProperties.createInstance(directConfig, CONFIG_SECTION_NAME);
        this.freelancerCardModerationService = freelancerCardModerationService;
        this.logbrokerClientFactoryService = logbrokerClientFactoryService;
        this.lbSeqIdService = lbSeqIdService;
    }

    @Override
    public void execute() {
        String jobEnabledPropValue = ppcPropertiesSupport.get(FREELANCER_CARDS_MODERATION_SENDING_ENABLED.getName());
        boolean jobEnabled = Boolean.parseBoolean(jobEnabledPropValue);
        if (!jobEnabled) {
            logger.info("Skip processing. Job is not enabled. {}={}",
                    FREELANCER_CARDS_MODERATION_SENDING_ENABLED,
                    jobEnabledPropValue);
            return;
        }
        sendRequests();
    }

    private void sendRequests() {
        int shard = getShard();
        List<CardModerationRequest> requests = freelancerCardModerationService.getRequests(shard);
        logger.info("Requests were found: {}", requests.size());
        if (requests.isEmpty()) {
            return;
        }
        int counter = 0;
        String sourceId = getSourceId(shard);
        try (AsyncProducer asyncProducer = logbrokerClientFactoryService
                .createProducer(producerProperties, sourceId)) {
            Supplier<Long> seqIdSupplier = getSeqIdSupplier(shard);
            RequestSender requestSender = new RequestSender(asyncProducer, seqIdSupplier);
            for (CardModerationRequest request : requests) {
                requestSender.send(request);
                Long freelancerCardId = request.getMeta().getFreelancerCardId();
                freelancerCardModerationService.markCardAsInProgress(shard, freelancerCardId);
                counter++;
            }
        } catch (RuntimeException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException("Error while sending a FreelancerCard moderation request to LogBroker", ex);
        } finally {
            logger.info("Requests were sent: {}", counter);
        }
    }

    private Supplier<Long> getSeqIdSupplier(int shard) {
        return () -> lbSeqIdService.getNextCardModerationSeqId(shard);
    }

    private String getSourceId(int shard) {
        String className = this.getClass().getSimpleName();
        String shardAsStr = Integer.toString(shard);
        return String.join(":", className, shardAsStr);
    }
}
