package ru.yandex.direct.jobs.freelancers.moderation.sending;

import java.nio.charset.StandardCharsets;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import ru.yandex.direct.jobs.freelancers.moderation.sending.model.CardModerationRequest;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.transport.message.inbound.ProducerWriteResponse;


/**
 * Класс реализующий отправку заявок на модерацию карточек фрилансеров в Модерацию через топик в лог-брокере.
 */
class RequestSender {
    /**
     * Время за которое запрос должен был гарантированно отправиться в логброкер.
     * Взято с потолка, но с запасом, если за это время не отправилось, то по любому что-то пошло не так.
     */
    private static final int SENDING_TIMEOUT_SEC = 60;

    private final AsyncProducer asyncProducer;
    private final Supplier<Long> seqIdSupplier;

    RequestSender(AsyncProducer asyncProducer, Supplier<Long> seqIdSupplier) {
        this.asyncProducer = asyncProducer;
        this.seqIdSupplier = seqIdSupplier;
    }

    public void send(CardModerationRequest request)
            throws ExecutionException, TimeoutException {
        byte[] binaryData = convertToByteArray(request);
        long shardSeqId = seqIdSupplier.get();
        CompletableFuture<ProducerWriteResponse> completableFuture = asyncProducer.write(binaryData, shardSeqId);
        futureComplete(completableFuture);
    }

    private void futureComplete(CompletableFuture<ProducerWriteResponse> completableFuture)
            throws ExecutionException, TimeoutException {
        try {
            completableFuture.get(SENDING_TIMEOUT_SEC, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(e);
        }
    }

    private byte[] convertToByteArray(CardModerationRequest request) {
        String jsonRequest = JsonUtils.toJson(request);
        return jsonRequest.getBytes(StandardCharsets.UTF_8);
    }

}
