package ru.yandex.direct.jobs.grut.watchlog

import com.google.common.util.concurrent.ThreadFactoryBuilder
import java.time.Duration
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ExecutorService
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.core.configuration.CoreConfiguration.GRUT_CLIENT_FOR_WATCHLOG
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.env.TypicalEnvironmentNoDev7
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl
import ru.yandex.direct.jobs.grut.watchlog.components.GrutWatchlogLogbrokerWriter
import ru.yandex.direct.jobs.grut.watchlog.components.processors.BannerCandidateWatchlogProcessor
import ru.yandex.direct.jobs.grut.watchlog.components.processors.CampaignWatchlogProcessor
import ru.yandex.direct.jobs.grut.watchlog.components.processors.WatchlogProcessor
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.HourglassDaemon
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmService
import ru.yandex.grut.client.GrutClient
import ru.yandex.kikimr.persqueue.auth.Credentials
import ru.yandex.kikimr.persqueue.compression.CompressionCodec

/**
 * Первая версия Watchlogbroker'а, документация будет чуть позже
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 15), needCheck = NonDevelopmentEnvironment::class,
    tags = [CheckTag.DIRECT_PRIORITY_1_NOT_READY],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DIMITROVSD],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT]
    )]
)
@HourglassDaemon(runPeriod = 5)
@Hourglass(periodInSeconds = 5, needSchedule = TypicalEnvironmentNoDev7::class)
class GrutWatchlogbrokerJob(
    tvmIntegration: TvmIntegration,
    directConfig: DirectConfig,
    ppcPropertiesSupport: PpcPropertiesSupport,
    shardHelper: ShardHelper,
    @Qualifier(GRUT_CLIENT_FOR_WATCHLOG) grutClientForWatchlog: GrutClient,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(GrutWatchlogbrokerJob::class.java)
    }

    private val logbrokerClientFactory = LogbrokerClientFactoryFacade {
        val serviceTicket = tvmIntegration.getTicket(TvmService.LOGBROKER_PRODUCTION)
        Credentials.tvm(serviceTicket)
    }

    private val watchlogProcessors: List<WatchlogProcessor>

    private val executor: ExecutorService

    private val watchlogbrokerLogbrokerConfig: WatchlogbrokerLogbrokerConfig
    private val logbrokerWriterByShard = ConcurrentHashMap<Int, GrutWatchlogLogbrokerWriter>()

    init {
        val config = directConfig.getBranch("watchlogbroker")
        val logbrokerConfig = config.getBranch("logbroker")

        watchlogbrokerLogbrokerConfig = WatchlogbrokerLogbrokerConfig(
            host = logbrokerConfig.getString("host"),
            retries = logbrokerConfig.getInt("retries"),
            timeoutSec = logbrokerConfig.getLong("timeout_sec"),
            writeTopic = logbrokerConfig.getString("write_topic"),
        )

        val watchlogConsumerPrefix = config.getString("watchlog_consumer_prefix")
        watchlogProcessors = listOf(
            BannerCandidateWatchlogProcessor(
                ppcPropertiesSupport,
                grutClientForWatchlog,
                { shard -> getLogbrokerWriter(shard) },
                watchlogConsumerPrefix,
                shardHelper,
            ),
            CampaignWatchlogProcessor(
                ppcPropertiesSupport,
                grutClientForWatchlog,
                { shard -> getLogbrokerWriter(shard) },
                watchlogConsumerPrefix,
                shardHelper,
            ),
        )
        executor = Executors.newFixedThreadPool(watchlogProcessors.size,
            ThreadFactoryBuilder().setNameFormat("watchlogbroker-%d").build())
    }

    override fun execute() {
        val processorFutures = watchlogProcessors
            .map { CompletableFuture.runAsync(Trace.current().wrap { it.run() }, executor) }
            .toTypedArray()

        val finalCompletableFuture = CompletableFuture.allOf(*processorFutures)

        try {
            finalCompletableFuture.get(5, TimeUnit.MINUTES)
        } catch (ex: Exception) {
            logger.error("Watchlogbroker iteration failed", ex)
            throw ex
        }
    }

    private fun getLogbrokerWriter(shard: Int): GrutWatchlogLogbrokerWriter {
        return logbrokerWriterByShard.computeIfAbsent(shard) { createLogbrokerWriter(shard) }
    }

    private fun createLogbrokerWriter(shard: Int): GrutWatchlogLogbrokerWriter {
        val logbrokerProducerProperties = LogbrokerProducerPropertiesImpl.newBuilder()
            .setHost(watchlogbrokerLogbrokerConfig.host)
            .setTimeoutSec(watchlogbrokerLogbrokerConfig.timeoutSec)
            .setRetries(watchlogbrokerLogbrokerConfig.retries)
            .setGroup(shard)
            .setWriteTopic(watchlogbrokerLogbrokerConfig.writeTopic)
            .setCompressionCodec(CompressionCodec.GZIP)
            .build()

        val producerSupplier =
            logbrokerClientFactory.createProducerSupplier(logbrokerProducerProperties, "GrutWatchlogbrokerJob-$shard")

        return GrutWatchlogLogbrokerWriter(
            producerSupplier,
            Duration.ofSeconds(logbrokerProducerProperties.timeoutSec),
            logbrokerProducerProperties.retries
        )
    }

    override fun finish() {
        executor.shutdown()
        logbrokerWriterByShard.values.forEach { it.close() }
    }
}

data class WatchlogbrokerLogbrokerConfig(
    val host: String,
    val retries: Int,
    val timeoutSec: Long,
    val writeTopic: String,
)
