package ru.yandex.direct.jobs.grut.watchlog.components.processors

import java.time.Duration
import org.slf4j.LoggerFactory
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyName
import ru.yandex.direct.ess.router.models.TEssEvent
import ru.yandex.direct.jobs.grut.watchlog.components.GrutWatchlogLogbrokerWriter
import ru.yandex.direct.jobs.grut.watchlog.components.enrichers.EventEnricher
import ru.yandex.direct.solomon.SolomonUtils
import ru.yandex.direct.tracing.Trace
import ru.yandex.grut.objectwatcher.ObjectWatcher
import ru.yandex.grut.objects.proto.client.Schema.EObjectType
import ru.yandex.grut.watchlog.Watch.TEvent
import ru.yandex.inside.yt.kosher.common.YtTimestamp
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.primitives.Counter
import ru.yandex.monlib.metrics.primitives.GaugeInt64

data class Metrics(
    val processorDelay: GaugeInt64,
    val eventRead: Counter,
    val eventSkipped: Counter,
    val eventProcessed: Counter,
)

abstract class WatchlogProcessor(
    ppcPropertiesSupport: PpcPropertiesSupport,
    objectWatcher: ObjectWatcher,
    private val getLogrokerWriterByShard: (Int) -> GrutWatchlogLogbrokerWriter,
    private val watchlogConsumerPrefix: String
) {

    private val logger = LoggerFactory.getLogger(javaClass)

    private val processorEnabledProp by lazy {
        ppcPropertiesSupport.get(getProcessorPropertyName(), Duration.ofMinutes(5))
    }
    private val watch by lazy {
        val watchlogConsumerName = watchlogConsumerPrefix + getObjectTypeName()
        objectWatcher.createWatch(watchlogConsumerName, getObjectType(), eventCountLimit())
    }
    private val metrics by lazy {
        val labels = Labels.of("object_type", getObjectTypeName())
        val processorDelay = SolomonUtils.SOLOMON_REGISTRY.gaugeInt64("processor_delay", labels)
        val eventRead = SolomonUtils.SOLOMON_REGISTRY.counter("watchlog_event_read", labels)
        val eventSkipped = SolomonUtils.SOLOMON_REGISTRY.counter("watchlog_event_skipped", labels)
        val eventProcessed = SolomonUtils.SOLOMON_REGISTRY.counter("watchlog_event_processed", labels)

        Metrics(processorDelay, eventRead, eventSkipped, eventProcessed)
    }

    fun run() = Trace.current().profile("watchlog_processor:${javaClass.name}").use {
        if (!processorEnabledProp.getOrDefault(false)) {
            logger.info("Processor is disabled, skip")
            return
        }

        val events = watch.getEventsImmediately()
        updateDelay(events)
        if (events.isEmpty()) {
            logger.info("No events, skip")
            return
        }

        logger.info("Got ${events.size} from watchlog")
        metrics.eventRead.add(events.size.toLong())

        val enrichedEvents = enrichEvents(events)
        val filteredEnrichedEvents = filterEvents(enrichedEvents)
        writeEvents(filteredEnrichedEvents)
        watch.commitOffset()
    }

    private fun updateDelay(events: List<TEvent>) {
        if (events.isEmpty()) {
            metrics.processorDelay.set(0)
            return
        }

        val lastTimestampSec = YtTimestamp.valueOf(events.last().timestamp).instant.epochSecond
        logger.info("Last timestamp is $lastTimestampSec")
        metrics.processorDelay.set(System.currentTimeMillis() / 1000 - lastTimestampSec)
    }

    fun enrichEvents(events: List<TEvent>): List<TEssEvent> {
        val objectType = getObjectType()
        val essEvents = events.map {
            TEssEvent.newBuilder().apply {
                event = it
                this.objectType = objectType
            }.build()
        }

        return enricherChain.enrichChain(essEvents)
    }

    private fun filterEvents(enrichedEvents: List<TEssEvent>): List<TEssEvent> {
        val enrichedEventsWithShard = enrichedEvents.asSequence()
            .filter { it.event.transactionContext.shard != 0 }
            .toList()

        val filteredEventsCount = enrichedEvents.size - enrichedEventsWithShard.size
        if (filteredEventsCount != 0) {
            logger.warn("Filtered $filteredEventsCount events")
            metrics.eventSkipped.add(filteredEventsCount.toLong())
        }

        return enrichedEventsWithShard
    }

    private fun writeEvents(enrichedEvents: List<TEssEvent>) {
        enrichedEvents.groupBy { it.event.transactionContext.shard }.forEach { (shard, events) ->
            writeEvents(shard, events)
        }
        metrics.eventProcessed.add(enrichedEvents.size.toLong())
    }

    private fun writeEvents(shard: Int, enrichedEvents: List<TEssEvent>) {
        logger.info("Writing ${enrichedEvents.size} events to shard $shard")
        getLogrokerWriterByShard(shard).write(enrichedEvents)
    }

    private fun getObjectTypeName(): String = getObjectType().name.lowercase()

    abstract val enricherChain: EventEnricher
    abstract fun getObjectType(): EObjectType
    abstract fun getProcessorPropertyName(): PpcPropertyName<Boolean>
    abstract fun eventCountLimit(): ULong
}
