package ru.yandex.direct.jobs.hypergeo

import com.google.common.collect.Lists
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoSegmentRepository
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.HourglassStretchPeriod
import ru.yandex.direct.scheduler.support.DirectShardedJob

/**
 * Джоба для удаления неиспользуемых гипер гео (как самих условий ретаргетинга/целей, так и сегментов Аудиторий).
 * Аналогична DeleteUnusedHyperGeosOneshot.
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(days = 2),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.DIRECT_PRIORITY_2],
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_DLYANGE],
        method = [NotificationMethod.EMAIL],
        status = [JugglerStatus.OK, JugglerStatus.CRIT]
    )]
)
// Запускается ежедневно в 01:00
@Hourglass(cronExpression = "0 0 1 * * ?", needSchedule = ProductionOnly::class)
@HourglassStretchPeriod
class DeleteUnusedHyperGeosJob @Autowired constructor(
    private val hyperGeoService: HyperGeoService,
    private val hyperGeoRepository: HyperGeoRepository,
    private val hyperGeoSegmentRepository: HyperGeoSegmentRepository,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : DirectShardedJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(DeleteUnusedHyperGeosJob::class.java)

        // сколько объектов удалится за одну итерацию внутри джобы
        private const val IDS_TO_DELETE_CHUNK = 1_000

        // лимит на число удаляемых объектов на шарде
        private const val DEFAULT_LIMIT = 4_000
    }

    override fun execute() {
        val idsLimitProperty: PpcProperty<Int> =
            ppcPropertiesSupport.get(PpcPropertyNames.HYPER_GEO_IDS_TO_DELETE_LIMIT)
        val limit = idsLimitProperty.getOrDefault(DEFAULT_LIMIT)

        deleteUnusedHyperGeos(shard, limit)
        deleteUnusedHyperGeoSegments(shard, limit)
    }

    // удалить гипер гео (условия ретаргетинга + цели) и сегменты (в Аудиториях и в hypergeo_segments)
    private fun deleteUnusedHyperGeos(shard: Int, limit: Int) {
        logger.info("shard $shard: start deleting unused hyper geos and their segments")
        val hyperGeoIds = hyperGeoRepository.getUnusedHyperGeoIds(shard, limit)
        logger.info("shard $shard: ${hyperGeoIds.size} unused hyper geos")
        for (hyperGeoIdsChunk in Lists.partition(hyperGeoIds.toList(), IDS_TO_DELETE_CHUNK)) {
            logAndDeleteHyperGeos(shard, hyperGeoIdsChunk)
        }
        logger.info("shard $shard: finished deleting unused hyper geos and their segments")
    }

    private fun logAndDeleteHyperGeos(shard: Int, hyperGeoIds: List<Long>) {
        logger.info("shard $shard: trying to delete $hyperGeoIds hyper geos and their segments")
        val hyperGeosData = hyperGeoRepository.getHyperGeosToLog(shard, hyperGeoIds)
        hyperGeosData.forEach { logger.info("Trying to delete hyper geo $it") }

        hyperGeoService.deleteHyperGeos(shard, null, hyperGeoIds)

        val notDeletedHyperGeoIds = hyperGeoRepository.getHyperGeosToLog(shard, hyperGeoIds)
            .map { it.id }
            .toSet()
        val deletedHyperGeoIds = hyperGeoIds.minus(notDeletedHyperGeoIds)
        if (deletedHyperGeoIds.isNotEmpty()) {
            logger.info("shard $shard: successfully deleted $deletedHyperGeoIds hyper geos")
        }
        if (notDeletedHyperGeoIds.isNotEmpty()) {
            logger.info("shard $shard: unsuccessfully tried to delete $notDeletedHyperGeoIds hyper geos")
        }
    }

    // удалить сегменты (в Аудиториях и в hypergeo_segments)
    private fun deleteUnusedHyperGeoSegments(shard: Int, limit: Int) {
        val idsNotToDeleteProperty: PpcProperty<Set<Long>> =
            ppcPropertiesSupport.get(PpcPropertyNames.HYPER_GEO_SEGMENT_IDS_TO_BYPASS_DELETING)
        val idsNotToDelete = idsNotToDeleteProperty.getOrDefault(setOf())

        logger.info("shard $shard: start deleting unused hyper geo segments")
        val hyperGeoSegmentIds = hyperGeoSegmentRepository.getUnusedHyperGeoSegmentIds(shard, idsNotToDelete, limit)
        logger.info("shard $shard: ${hyperGeoSegmentIds.size} unused hyper geo segments")
        for (hyperGeoSegmentIdsChunk in Lists.partition(hyperGeoSegmentIds.toList(), IDS_TO_DELETE_CHUNK)) {
            logAndDeleteHyperGeoSegments(shard, hyperGeoSegmentIdsChunk)
        }
    }

    private fun logAndDeleteHyperGeoSegments(shard: Int, hyperGeoSegmentIds: List<Long>) {
        logger.info("shard $shard: trying to delete $hyperGeoSegmentIds hyper geo segments")
        val hyperGeosData = hyperGeoSegmentRepository.getHyperGeoSegmentsToLog(shard, hyperGeoSegmentIds)
        hyperGeosData.forEach { logger.info("Trying to delete hyper geo segment $it") }

        hyperGeoService.deleteHyperGeoSegments(shard, hyperGeoSegmentIds)

        val notDeletedSegmentIds = hyperGeoSegmentRepository.getHyperGeoSegmentsToLog(shard, hyperGeoSegmentIds)
            .map { it.id }
            .toSet()
        val deletedHyperGeoSegmentIds = hyperGeoSegmentIds.minus(notDeletedSegmentIds)
        if (deletedHyperGeoSegmentIds.isNotEmpty()) {
            logger.info("shard $shard: successfully deleted $deletedHyperGeoSegmentIds hyper geo segments")
        }
        if (notDeletedSegmentIds.isNotEmpty()) {
            logger.info("shard $shard: unsuccessfully tried to delete $notDeletedSegmentIds hyper geo segments")
        }
    }
}
