package ru.yandex.direct.jobs.idm;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.util.RelaxedWorker;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.model.ClientsRelation;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.ENABLE_CLEAR_OLD_LIMITED_SUPPORTS_JOB;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Удаление заблокированных ролей LIMITED_SUPPORT
 * <p>
 * Для пользователей, логины которых были заблокированы в Директе
 * при неудачной попытке удалить их через idm (из-за наличия связанных ролей support_for_client):
 * удалим роль в Директе, если больше нет связанных ролей support_for_client
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2, minutes = 5),
        needCheck = NonDevelopmentEnvironment.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_MAXLOG,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        ),
        tags = {DIRECT_PRIORITY_1})
@Hourglass(periodInSeconds = 60 * 60, needSchedule = NonDevelopmentEnvironment.class)
public class ClearOldLimitedSupportsJob extends DirectShardedJob {
    private static final int CHUNK_SIZE = 50;
    private static final RelaxedWorker RELAXED_WORKER = new RelaxedWorker(3.0);

    private static final Logger logger = LoggerFactory.getLogger(ClearOldLimitedSupportsJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ClientRepository clientRepository;
    private final RbacClientsRelations rbacClientsRelations;
    private final ClientService clientService;
    private final UserService userService;

    @Autowired
    public ClearOldLimitedSupportsJob(
            PpcPropertiesSupport ppcPropertiesSupport,
            ClientRepository clientRepository,
            RbacClientsRelations rbacClientsRelations,
            ClientService clientService,
            UserService userService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.clientRepository = clientRepository;
        this.rbacClientsRelations = rbacClientsRelations;
        this.clientService = clientService;
        this.userService = userService;
    }

    /**
     * Конструктор нужен только для тестов. Используется для указания шарда.
     */
    ClearOldLimitedSupportsJob(
            int shard,
            PpcPropertiesSupport ppcPropertiesSupport,
            ClientRepository clientRepository,
            RbacClientsRelations rbacClientsRelations,
            ClientService clientService,
            UserService userService) {
        super(shard);
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.clientRepository = clientRepository;
        this.rbacClientsRelations = rbacClientsRelations;
        this.clientService = clientService;
        this.userService = userService;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Skip processing. Job is not enabled.");
            return;
        }
        int shard = getShard();

        Set<ClientsRole> clientsRoles = Set.of(RbacRole.toSource(RbacRole.LIMITED_SUPPORT));
        List<Long> limitedSupportUids = clientRepository.getClientChiefUidsWithRoles(shard, clientsRoles);

        for (List<Long> chunk : Iterables.partition(limitedSupportUids, CHUNK_SIZE)) {
            logger.info("uids for update: {}", chunk);
            RELAXED_WORKER.runAndRelax(() -> processChunk(chunk));
        }
    }

    private boolean isJobEnabled() {
        return ppcPropertiesSupport.get(ENABLE_CLEAR_OLD_LIMITED_SUPPORTS_JOB).getOrDefault(false);
    }

    private void processChunk(List<Long> userIds) {
        Collection<User> users = userService.massGetUser(userIds);
        List<User> blockedLimitedSupports = StreamEx.of(users)
                .filter(User::getStatusBlocked)
                .filter(user -> user.getRole() == RbacRole.LIMITED_SUPPORT)
                .toList();

        List<ClientId> supportIds = mapList(blockedLimitedSupports, User::getClientId);
        List<ClientsRelation> allSupportRelations = rbacClientsRelations.getSupportRelationsByOperatorIds(supportIds);

        Map<ClientId, List<ClientsRelation>> relationsBySupportId = StreamEx.of(allSupportRelations)
                .groupingBy(relation -> ClientId.fromLong(relation.getClientIdFrom()));

        for (User support : blockedLimitedSupports) {
            List<ClientsRelation> relations = relationsBySupportId.get(support.getClientId());
            if (CollectionUtils.isEmpty(relations)) {
                removeRole(support);
            }
        }
    }

    private void removeRole(User support) {
        clientService.updateClientRole(support.getClientId(), RbacRole.EMPTY, null);
        userService.unblockUser(support.getClientId(), support.getUid());
        logger.info("remove_role uid: {}, login: {}", support.getUid(), support.getLogin());
    }
}
