package ru.yandex.direct.jobs.interceptors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.scheduler.JobInterceptor;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.WrappedJob;
import ru.yandex.direct.tracing.Trace;

/**
 * Логирование запуска и останова задачи
 */
@Component
@ParametersAreNonnullByDefault
public class JobLoggingInterceptor implements JobInterceptor {
    private static final Logger logger = LoggerFactory.getLogger(JobLoggingInterceptor.class);

    @Override
    @Nonnull
    public WrappedJob wrap(WrappedJob code, BaseDirectJob originalJob) {
        return () -> {
            String tags = Trace.current().getTags();
            String msgSuffix = Strings.isNullOrEmpty(tags) ? "" : " " + tags;
            boolean needLogging = needLogging(originalJob);
            if (needLogging) {
                logger.info("START" + msgSuffix);
            }
            boolean jobSuccess = false;
            try {
                code.run();
                jobSuccess = true;
            } catch (Exception e) {
                logger.error("Job " + originalJob.getClass().getSimpleName() + msgSuffix
                        + " threw Exception: ", e);
                throw e;
            } catch (Throwable e) {
                logger.error("Job " + originalJob.getClass().getSimpleName() + msgSuffix
                        + " threw Throwable: ", e);
                throw e;
            } finally {
                if (needLogging) {
                    logger.info((jobSuccess ? "FINISH" : "FAILED") + msgSuffix);
                }
            }
        };
    }

    private boolean needLogging(BaseDirectJob job) {
        return !job.getClass().isAnnotationPresent(IgnoreStartFinishLogging.class);
    }
}
