package ru.yandex.direct.jobs.interceptors;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.AtomicDouble;
import org.springframework.stereotype.Component;

import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.scheduler.JobInterceptor;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.WrappedJob;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Сбор статистики
 */
@Component
@ParametersAreNonnullByDefault
public class JobStatsInterceptor implements JobInterceptor {
    private final AtomicLong started = new AtomicLong(0);
    private final AtomicLong successes = new AtomicLong(0);
    private final AtomicLong fails = new AtomicLong(0);
    private final AtomicLong running = new AtomicLong(0);
    private final AtomicDouble time = new AtomicDouble(0);


    public JobStatsInterceptor(TaskThreadPool threadPool) {
        SOLOMON_REGISTRY.lazyCounter("hourglass_started", started::longValue);
        SOLOMON_REGISTRY.lazyCounter("hourglass_successes", successes::longValue);
        SOLOMON_REGISTRY.lazyCounter("hourglass_fails", fails::longValue);
        SOLOMON_REGISTRY.lazyCounter("hourglass_running", running::longValue);
        SOLOMON_REGISTRY.lazyCounter("hourglass_threads", threadPool::getPoolSize);
        SOLOMON_REGISTRY.lazyGaugeDouble("hourglass_time_seconds", time::doubleValue);

        SOLOMON_REGISTRY.lazyCounter("quartz_started", started::longValue);
        SOLOMON_REGISTRY.lazyCounter("quartz_successes", successes::longValue);
        SOLOMON_REGISTRY.lazyCounter("quartz_fails", fails::longValue);
        SOLOMON_REGISTRY.lazyCounter("quartz_running", running::longValue);
        SOLOMON_REGISTRY.lazyCounter("quartz_threads", threadPool::getPoolSize);
        SOLOMON_REGISTRY.lazyGaugeDouble("quartz_time_seconds", time::doubleValue);
    }

    @Override
    @Nonnull
    public WrappedJob wrap(WrappedJob code, BaseDirectJob originalJob) {
        return () -> {
            started.incrementAndGet();
            running.incrementAndGet();
            long startTime = System.nanoTime();
            boolean jobSuccess = false;
            try {
                code.run();
                jobSuccess = true;
            } finally {
                (jobSuccess ? successes : fails).incrementAndGet();
                running.decrementAndGet();
                double deltaSeconds = ((double) (System.nanoTime() - startTime)) / TimeUnit.SECONDS.toNanos(1);
                time.addAndGet(deltaSeconds);
            }
        };
    }
}
