package ru.yandex.direct.jobs.interceptors;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.scheduler.JobInterceptor;
import ru.yandex.direct.scheduler.support.BaseDirectJob;
import ru.yandex.direct.scheduler.support.WrappedJob;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;

import static ru.yandex.direct.tracing.InterceptorUtilKt.formatTags;

/**
 * Запуск трейсинга для задачи
 */
@Component
@ParametersAreNonnullByDefault
public class JobTracingInterceptor implements JobInterceptor {
    private final TraceHelper traceHelper;

    @Autowired
    public JobTracingInterceptor(TraceHelper traceHelper) {
        this.traceHelper = traceHelper;
    }

    @Override
    @Nonnull
    public WrappedJob wrap(WrappedJob job, BaseDirectJob originalJob) {
        String method = formatMethodName(originalJob);
        List<String> tags = formatTags(originalJob);

        return () -> {
            try (TraceGuard guard = traceHelper.guard(method, tags)) {
                job.run();
            }
        };
    }

    public static String formatMethodName(BaseDirectJob job) {
        return formatMethodName(job.getClass());
    }

    public static String formatMethodName(Class<?> clazz) {
        String[] packageParts = clazz.getPackage().getName().split("\\.");
        if (packageParts.length > 0) {
            return packageParts[packageParts.length - 1] + "." + clazz.getSimpleName();
        } else {
            return clazz.getSimpleName();
        }
    }
}
