package ru.yandex.direct.jobs.internal;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.internal.model.StructureOfBannerIds;
import ru.yandex.direct.jobs.internal.utils.InfoForUrlNotificationsGetter;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Включает баннеры внутренней рекламы, остановленные урл-мониторингом.
 * Джоба устроена следующим образом: выгружает баннеры, остановленные урл-мониторингом, и включат только те из них,
 * у которых все ссылки валидны. Проверка валидности ссылок осуществляется через Зору.
 * <p>
 * Используем cronExpression, чтобы запускаться на 2 часа позже после джобы {@link UpdateBannersUnreachableUrlJob}.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 14),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_XY6ER,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 14),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(cronExpression = "0 4 4/4 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class EnableBannersUnreachableUrlJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(EnableBannersUnreachableUrlJob.class);

    private final BannerTypedRepository bannerTypedRepository;
    private final DslContextProvider ppcDslContextProvider;
    private final BannerModifyRepository modifyRepository;
    private final BannerUrlCheckService bannerUrlCheckService;
    private final UrlMonitoringNotifyService urlMonitoringNotifyService;
    private final TemplateInfoService templateInfoService;
    private final InfoForUrlNotificationsGetter infoForUrlNotificationsGetter;

    @Autowired
    public EnableBannersUnreachableUrlJob(BannerTypedRepository bannerTypedRepository,
                                          DslContextProvider ppcDslContextProvider,
                                          BannerModifyRepository modifyRepository,
                                          BannerUrlCheckService bannerUrlCheckService,
                                          UrlMonitoringNotifyService urlMonitoringNotifyService,
                                          TemplateInfoService templateInfoService,
                                          InfoForUrlNotificationsGetter infoForUrlNotificationsGetter) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.modifyRepository = modifyRepository;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.urlMonitoringNotifyService = urlMonitoringNotifyService;
        this.templateInfoService = templateInfoService;
        this.infoForUrlNotificationsGetter = infoForUrlNotificationsGetter;
    }

    @Override
    public void execute() {
        List<InternalBanner> fetchedBanners =
                bannerTypedRepository.getNoArchivedInternalBannersStoppedByUrlMonitoring(getShard());
        logger.info("fetched {} banners  stopped by url monitoring: {}", fetchedBanners.size(),
                JsonUtils.toJson(fetchedBanners));

        Map<Long, Set<String>> urlsByBannerId = getUrlsByBannerId(fetchedBanners);

        List<Long> bannersToEnable = getBannersToEnable(urlsByBannerId);
        logger.info("found {} banners with all reachable urls: {}", bannersToEnable.size(),
                JsonUtils.toJson(bannersToEnable));

        Set<Long> enabledBannerIds = enableBanners(bannersToEnable);
        logger.info("enabled banners: {}", enabledBannerIds);

        Predicate<InternalBanner> isBannerEnabled = banner -> enabledBannerIds.contains(banner.getId());
        List<StructureOfBannerIds> structuresOfEnabledBanners = getStructuredAndFilteredBannerIds(fetchedBanners, isBannerEnabled);

        urlMonitoringNotifyService.notifyBannersEnabled(structuresOfEnabledBanners);
    }

    Map<Long, Set<String>> getUrlsByBannerId(List<InternalBanner> fetchedBanners) {
        Map<Long, Set<Long>> urlResourceIdsByTemplateId = getUrlResourceIdsByTemplateId(fetchedBanners);

        return StreamEx.of(fetchedBanners)
                .filter(banner -> urlResourceIdsByTemplateId.containsKey(banner.getTemplateId()))
                .mapToEntry(InternalBanner::getId, Function.identity())
                .mapValues(banner -> extractUrls(banner, urlResourceIdsByTemplateId.get(banner.getTemplateId())))
                .toMap();
    }

    Map<Long, Set<Long>> getUrlResourceIdsByTemplateId(List<InternalBanner> fetchedBanners) {
        Set<Long> templateIds = listToSet(fetchedBanners, InternalBanner::getTemplateId);

        Map<Long, InternalTemplateInfo> templateInfoByTemplateId =
                listToMap(templateInfoService.getByTemplateIds(templateIds), InternalTemplateInfo::getTemplateId);

        return EntryStream.of(templateInfoByTemplateId)
                .mapValues(InternalTemplateInfo::getResources)
                .flatMapValues(Collection::stream)
                .filterValues(resourceInfo -> resourceInfo.getType() == ResourceType.URL)
                .mapValues(ResourceInfo::getId)
                .grouping(Collectors.toSet());
    }

    Set<String> extractUrls(InternalBanner banner, Set<Long> urlResourceIds) {
        return StreamEx.of(banner.getTemplateVariables())
                .filter(templateVariable ->
                        urlResourceIds.contains(templateVariable.getTemplateResourceId())
                )
                .map(TemplateVariable::getInternalValue)
                .filter(Objects::nonNull)
                .toSet();
    }

    List<Long> getBannersToEnable(Map<Long, Set<String>> urlsByBannerId) {
        return EntryStream.of(urlsByBannerId)
                .filterValues(urls -> StreamEx.of(urls)
                        .allMatch(url -> bannerUrlCheckService.isUrlReachable(url).getResult())
                )
                .keys()
                .toList();
    }

    Set<Long> enableBanners(List<Long> bannerIdsToEnable) {
        Set<Long> enabledBanners = new HashSet<>();

        ppcDslContextProvider.ppcTransactionResult(getShard(), configuration -> {
            DSLContext dslContext = DSL.using(configuration);

            List<ModelChanges<BannerWithInternalInfo>> modelChanges = mapList(bannerIdsToEnable,
                    id -> new ModelChanges<>(id, BannerWithInternalInfo.class)
                            .process(true, BannerWithInternalInfo.STATUS_SHOW)
                            .process(StatusBsSynced.NO, BannerWithInternalInfo.STATUS_BS_SYNCED)
                            .process(false, BannerWithInternalInfo.IS_STOPPED_BY_URL_MONITORING)
            );

            return modifyRepository.updateByPredicate(
                    dslContext, new BannerRepositoryContainer(getShard()), modelChanges, BannerWithInternalInfo.class,
                    banner -> {
                        if (!banner.getIsStoppedByUrlMonitoring()) {
                            return false;
                        }
                        enabledBanners.add(banner.getId());
                        return true;
                    }
            );
        });
        return enabledBanners;
    }

    List<StructureOfBannerIds> getStructuredAndFilteredBannerIds(List<InternalBanner> fetchedBanners,
                                                           Predicate<InternalBanner> filter) {
        var filteredBanners = StreamEx.of(fetchedBanners).filter(filter).toList();

        return infoForUrlNotificationsGetter.getAdditionalInfoByInternalBanners(getShard(), filteredBanners);
    }
}
