package ru.yandex.direct.jobs.internal;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.internalads.model.CryptaExportType;
import ru.yandex.direct.core.entity.internalads.model.CryptaSegment;
import ru.yandex.direct.core.entity.internalads.repository.InternalCryptaSegmentsYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope;
import ru.yandex.direct.core.entity.retargeting.model.CryptaInterestType;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.Counter;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.db.PpcPropertyNames.CRYPTA_SEGMENTS_LAST_IMPORT_DATE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;

/**
 * Импортирует в таблицу {@link Tables#CRYPTA_GOALS} новые сегменты для внутренней рекламы из YT таблицы:
 * //home/crypta/production/profiles/export/lab/segments
 * Джоба добавляет только записи! Удаление или обновление не делается
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,

        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_XY6ER,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 2 * 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class ImportCryptaSegmentsJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(ImportCryptaSegmentsJob.class);

    // Крипта выгружает сегменты только на Хану. Для внутренней рекламы это норм
    static final YtCluster YT_CLUSTER = YtCluster.HAHN;
    private static final String TANKER_NAME_PREFIX = "crypta_";
    private static final String TANKER_NAME_SUFFIX = "_name";

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final InternalCryptaSegmentsYtRepository ytRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;

    @Autowired
    public ImportCryptaSegmentsJob(PpcPropertiesSupport ppcPropertiesSupport,
                                   InternalCryptaSegmentsYtRepository ytRepository,
                                   CryptaSegmentRepository cryptaSegmentRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytRepository = ytRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
    }

    @Override
    public void execute() {
        LocalDate ytTableGenerateDate = ytRepository.getTableGenerateDate(YT_CLUSTER);
        PpcProperty<LocalDate> lastImportDateProperty = ppcPropertiesSupport.get(CRYPTA_SEGMENTS_LAST_IMPORT_DATE);
        if (!needImport(ytTableGenerateDate, lastImportDateProperty)) {
            logger.info("segments table is already imported");
            return;
        }

        List<CryptaSegment> ytFetchedRecords = ytRepository.getAll(YT_CLUSTER);
        logger.info("fetched {} records from YT", ytFetchedRecords.size());
        checkState(!ytFetchedRecords.isEmpty(), "fetched records from YT can't be empty");

        List<Goal> mysqlFetchedRecords =
                List.copyOf(cryptaSegmentRepository.getAll(CryptaGoalScope.INTERNAL_AD).values());
        logger.info("fetched {} records from MySQL", mysqlFetchedRecords.size());

        List<Goal> newRecordsToAdd = getNewRecordsToAdd(ytFetchedRecords, mysqlFetchedRecords);
        int addedRecordsCount = cryptaSegmentRepository.add(newRecordsToAdd);
        logger.info("added {} new records", addedRecordsCount);
        checkState(addedRecordsCount == newRecordsToAdd.size(),
                "addedRecordsCount is not equal to number of new records=%s", newRecordsToAdd.size());

        logger.info("setting {} property value to '{}'",
                CRYPTA_SEGMENTS_LAST_IMPORT_DATE.getName(), ytTableGenerateDate);
        lastImportDateProperty.set(ytTableGenerateDate);
    }

    /**
     * Нужно ли обновлять MySQL таблицу {@link Tables#CRYPTA_GOALS}
     *
     * @param ytTableGenerateDate    время генерации YT таблицы {@link YtDbTables#CRYPTASEGMENTS}
     * @param lastImportDateProperty проперти, которая хранит время последнего импорта
     * @return true/false
     */
    static boolean needImport(LocalDate ytTableGenerateDate, PpcProperty<LocalDate> lastImportDateProperty) {
        LocalDate lastImportDate = lastImportDateProperty.get();
        logger.info("{} property value is '{}'. generate_date from YT is '{}'",
                CRYPTA_SEGMENTS_LAST_IMPORT_DATE.getName(), lastImportDate, ytTableGenerateDate);

        return lastImportDate == null || lastImportDate.isBefore(ytTableGenerateDate);
    }

    /**
     * Возвращает записи, которых нет в базе MySQL, но есть в YT'е
     */
    static List<Goal> getNewRecordsToAdd(List<CryptaSegment> ytFetchedRecords, List<Goal> mysqlFetchedRecords) {
        Map<CryptaSegmentDescriptor, CryptaSegment> ytSegmentByDescriptor = getSegmentWithGroups(ytFetchedRecords);
        Map<CryptaSegmentDescriptor, Goal> mysqlGoalByDescriptor = StreamEx.of(mysqlFetchedRecords)
                .mapToEntry(ImportCryptaSegmentsJob::toDescriptor, identity())
                .distinctKeys()
                .toMap();

        Map<CryptaSegmentDescriptor, Goal> newRecordsToAdd = getNewGoals(ytSegmentByDescriptor, mysqlGoalByDescriptor);
        long lastGoalIdFromDb = StreamEx.of(mysqlFetchedRecords)
                .mapToLong(Goal::getId)
                .max()
                .orElseThrow(() -> new IllegalStateException("got empty internal_ad goals from mysql"));
        setGoalIdAndParentId(lastGoalIdFromDb, newRecordsToAdd, ytSegmentByDescriptor, mysqlGoalByDescriptor);

        return List.copyOf(newRecordsToAdd.values());
    }

    /**
     * Возвращает мапу c сегментами полученными из YT'я и с добавлением групп по которым нет отдельной записи в YT'е
     *
     * @see #getPseudoGroups(Map)
     * @see #getRootGroups(Map)
     */
    private static Map<CryptaSegmentDescriptor, CryptaSegment> getSegmentWithGroups(List<CryptaSegment> ytFetchedRecords) {
        var segmentByDescriptor = listToMap(ytFetchedRecords, ImportCryptaSegmentsJob::toDescriptor);

        return EntryStream.of(segmentByDescriptor)
                .append(getPseudoGroups(segmentByDescriptor))
                .append(getRootGroups(segmentByDescriptor))
                .toMap();
    }

    /**
     * Возвращает мапу с сегментами для псевдо-групп
     * Псевдо-группа это группа, у которого есть сегменты. Для псевдо-групп в YT таблице нет отдельной записи, поэтому
     * добавляем запись в мапу, чтобы в mysql сохранили отдельную запись. По факту в мапу добавляем Descriptor для
     * псевдо-группы и первый попавшийся сегмент псевдо-группы
     * Пример псевдо-группы: https://lab.crypta.yandex-team.ru/segments?group=root-crypta&segment=segment-pets
     */
    private static Map<CryptaSegmentDescriptor, CryptaSegment> getPseudoGroups(
            Map<CryptaSegmentDescriptor, CryptaSegment> ytSegmentByDescriptor) {
        Set<String> parentIds =
                mapAndFilterToSet(ytSegmentByDescriptor.values(), CryptaSegment::getParentId, Objects::nonNull);

        return StreamEx.ofValues(ytSegmentByDescriptor)
                .mapToEntry(CryptaSegment::getId, identity())
                .filterKeys(parentIds::contains)
                .distinctKeys()
                .mapKeys(CryptaSegmentDescriptor::createForGroup)
                .removeKeys(ytSegmentByDescriptor::containsKey)
                .toMap();
    }

    /**
     * Возвращает мапу с сегментами для root-групп
     * Root-группа — самый верхний узел дерева, в YT таблице нет отдельной записи, поэтому добавляем запись в мапу,
     * чтобы в mysql сохранили отдельную запись
     * Пример root-группы: https://lab.crypta.yandex-team.ru/segments?group=root-crypta&segment=root-crypta
     */
    private static Map<CryptaSegmentDescriptor, CryptaSegment> getRootGroups(
            Map<CryptaSegmentDescriptor, CryptaSegment> ytSegmentByDescriptor) {
        Set<String> parentIds =
                mapAndFilterToSet(ytSegmentByDescriptor.values(), CryptaSegment::getParentId, Objects::nonNull);
        Map<String, CryptaSegment> ytSegmentById = EntryStream.of(ytSegmentByDescriptor)
                .mapKeys(CryptaSegmentDescriptor::getCryptaId)
                .distinctKeys()
                .toMap();

        return StreamEx.of(parentIds)
                .remove(ytSegmentById::containsKey)
                .map(CryptaSegmentDescriptor::createForGroup)
                .mapToEntry(ImportCryptaSegmentsJob::getSegmentForRootGroup)
                .toMap();
    }

    private static CryptaSegment getSegmentForRootGroup(CryptaSegmentDescriptor descriptor) {
        return new CryptaSegment()
                .withId(descriptor.getCryptaId())
                .withParentId("")
                .withName(descriptor.getCryptaId());
    }

    /**
     * Возвращает новые цели, которых нет в базе MySQL
     */
    private static Map<CryptaSegmentDescriptor, Goal> getNewGoals(
            Map<CryptaSegmentDescriptor, CryptaSegment> ytSegmentByDescriptor,
            Map<CryptaSegmentDescriptor, Goal> mysqlGoalByDescriptor) {
        var difference = SetUtils.difference(ytSegmentByDescriptor.keySet(), mysqlGoalByDescriptor.keySet());

        return EntryStream.of(ytSegmentByDescriptor)
                .filterKeys(difference::contains)
                .mapToValue(ImportCryptaSegmentsJob::toGoal)
                .toMap();
    }

    /**
     * Для новых целей проставляет GoalId и ParentId
     * Сначала для новых целей по порядку проставляется {@code goalId = lastGoalIdFromDb + counter.next()}
     * Потом для этих целей проставляется ParentId, сначала ищем родительскую цель среди целей полученных из MySQL,
     * если нет - то берем из целей, полученных из YT'я
     */
    private static void setGoalIdAndParentId(long lastGoalIdFromDb,
                                             Map<CryptaSegmentDescriptor, Goal> newGoalByDescriptor,
                                             Map<CryptaSegmentDescriptor, CryptaSegment> ytSegmentByDescriptor,
                                             Map<CryptaSegmentDescriptor, Goal> mysqlGoalByDescriptor) {
        var counter = new Counter(1);
        newGoalByDescriptor.forEach((ignore, goal) -> goal.setId(lastGoalIdFromDb + counter.next()));
        long lastId = lastGoalIdFromDb + counter.next() - 1;
        checkState(lastId < Goal.CRYPTA_INTERNAL_UPPER_BOUND,
                "lastId=%s of new goals greater than max allowed id for internal_ad goals: %s",
                lastId, Goal.CRYPTA_INTERNAL_UPPER_BOUND);

        EntryStream.of(newGoalByDescriptor)
                .forKeyValue((descriptor, goal) -> {
                    String parentId = ytSegmentByDescriptor.get(descriptor).getParentId();
                    if (StringUtils.isEmpty(parentId)) {
                        goal.setParentId(0L);
                        return;
                    }

                    CryptaSegmentDescriptor parentDescriptor = CryptaSegmentDescriptor.createForGroup(parentId);
                    Goal parentGoal = mysqlGoalByDescriptor
                            .getOrDefault(parentDescriptor, newGoalByDescriptor.get(parentDescriptor));
                    checkState(parentGoal != null, "not found parentGoal with id=%s", parentId);

                    goal.setParentId(parentGoal.getId());
                });
    }

    private static Goal toGoal(CryptaSegmentDescriptor descriptor, CryptaSegment cryptaSegment) {
        return (Goal) new Goal()
                .withKeyword(descriptor.getKeywordId())
                .withKeywordValue(descriptor.getSegmentId())
                .withTankerNameKey(String.join("", TANKER_NAME_PREFIX, descriptor.getCryptaId(), TANKER_NAME_SUFFIX))
                .withTankerDescriptionKey(String.format("%s%s_description",
                        TANKER_NAME_PREFIX, descriptor.getCryptaId()))
                .withName(cryptaSegment.getName())
                .withInterestType(toCryptaInterestType(cryptaSegment.getExportType()))
                .withType(GoalType.INTERNAL)
                .withCryptaScope(Set.of(ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope.INTERNAL_AD))
                .withKeywordShort("")
                .withKeywordValueShort("");
    }

    private static CryptaInterestType toCryptaInterestType(@Nullable CryptaExportType exportType) {
        if (exportType == CryptaExportType.SHORTTERM) {
            return CryptaInterestType.short_term;
        } else if (exportType == CryptaExportType.LONGTERM) {
            return CryptaInterestType.short_term;
        }

        return CryptaInterestType.all;
    }

    /**
     * Возвращает дескриптор для сегмента
     */
    private static CryptaSegmentDescriptor toDescriptor(CryptaSegment cryptaSegment) {
        String keywordId = cryptaSegment.getExportKeywordId() != null
                ? cryptaSegment.getExportKeywordId().toString()
                : "0";
        String segmentId = cryptaSegment.getExportSegmentId() != null
                ? cryptaSegment.getExportSegmentId().toString()
                : "0";
        return CryptaSegmentDescriptor.create(cryptaSegment.getId(), keywordId, segmentId);
    }

    /**
     * Возвращает дескриптор для цели
     */
    static CryptaSegmentDescriptor toDescriptor(Goal goal) {
        String cryptaId = goal.getTankerNameKey()
                .replaceAll("^" + TANKER_NAME_PREFIX, "")
                .replaceAll(TANKER_NAME_SUFFIX + "$", "");
        return CryptaSegmentDescriptor.create(cryptaId, goal.getKeyword(), goal.getKeywordValue());
    }

}
