package ru.yandex.direct.jobs.internal

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.jobs.internal.model.BannerModerationStuckMessage
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.mail.MailSender
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.HourglassStretchPeriod
import ru.yandex.direct.scheduler.support.DirectShardedJob
import java.time.LocalDateTime


/**
 * Джоба для проверки "застрявших" на модерации банеров внутренней рекламы.
 * Рассылает уведомления об id банера и его текущем статусе.
 */
@JugglerCheck(
    needCheck = ProductionOnly::class,
    ttl = JugglerCheck.Duration(days = 2, hours = 8),
    tags = [CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM],
    notifications = [
        OnChangeNotification(
            recipient = [NotificationRecipient.LOGIN_XY6ER],
            method = [NotificationMethod.TELEGRAM]
        )
    ]
)
@Hourglass(cronExpression = "0 0 9-17/4 * * ?", needSchedule = TypicalEnvironment::class)
@HourglassStretchPeriod
class InternalBannerModerationStuckNotificationJob(
    val bannerTypedRepository: BannerTypedRepository,
    val mailSender: MailSender
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(InternalBannerModerationStuckNotificationJob::class.java)
        const val MAX_AGE_HOURS: Long = 4
    }

    override fun execute() {
        val dateTime = LocalDateTime.now().minusHours(MAX_AGE_HOURS)
        val banners = bannerTypedRepository.getInternalBannersOnModerationOlderThan(shard, dateTime)

        if (banners.isNotEmpty()) {
            val mailMessage = BannerModerationStuckMessage(banners, MAX_AGE_HOURS)

            logger.info("Mail body: ${mailMessage.messageBody}");
            mailSender.send(mailMessage)
        }
    }

}
