package ru.yandex.direct.jobs.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.EnumUtils;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.internalads.model.BannerUnreachableUrl;
import ru.yandex.direct.core.entity.internalads.repository.BannersUnreachableUrlYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.core.service.urlchecker.UrlCheckResult;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.jobs.internal.model.StructureOfBannerIds;
import ru.yandex.direct.jobs.internal.model.StructureOfUnavailableBanners;
import ru.yandex.direct.jobs.internal.utils.InfoForUrlNotificationsGetter;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.db.PpcPropertyNames.BANNERS_UNREACHABLE_URL_ERRORS_NOT_DISABLE;
import static ru.yandex.direct.common.db.PpcPropertyNames.BANNERS_UNREACHABLE_URL_IDS_NOT_DISABLE;
import static ru.yandex.direct.common.db.PpcPropertyNames.BANNERS_UNREACHABLE_URL_STOP_BANNERS_ENABLED;
import static ru.yandex.direct.common.db.PpcPropertyNames.BANNERS_UNREACHABLE_URL_URLS_NOT_DISABLE;
import static ru.yandex.direct.core.entity.internalads.repository.BannersUnreachableUrlYtRepository.createBannerUnreachableUrl;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Отключает баннеры внутренней рекламы с недоступными ссылками.
 * Источником данных является YT таблица: {@link YtDbTables#BANNERSUNREACHABLEURL}. Таблица заполняется
 * <a href="https://sandbox.yandex-team.ru/scheduler/8503/view">таской</a>, прочитать подробнее про нее можно
 * <a href="https://wiki.yandex-team.ru/adv-interfaces/direct/banana/#validacijassylok">тут</a>.
 * <p>
 * Джоба устроена следующим образом: делает выгрузку данных из yt-таблицы
 * <a href="https://yt.yandex-team.ru/hahn/navigation?path=//home/yabs/banana/WrongDirectLinks">WrongDirectLinks</a>,
 * простукивает через Зору выгруженные ссылки и останавливает те баннеры, у которых урлы все еще недоступны.
 * <p>
 * Используем cronExpression, чтобы запускаться сразу после таски.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 14),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_XY6ER,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 14),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(cronExpression = "0 4 2/4 * * ?", needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateBannersUnreachableUrlJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(UpdateBannersUnreachableUrlJob.class);

    private final ShardHelper shardHelper;
    private final DslContextProvider ppcDslContextProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final BannersUnreachableUrlYtRepository ytRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerUrlCheckService bannerUrlCheckService;
    private final UrlMonitoringNotifyService urlMonitoringNotifyService;
    private final InfoForUrlNotificationsGetter infoForUrlNotificationsGetter;

    @Autowired
    public UpdateBannersUnreachableUrlJob(ShardHelper shardHelper,
                                          DslContextProvider ppcDslContextProvider,
                                          PpcPropertiesSupport ppcPropertiesSupport,
                                          BannersUnreachableUrlYtRepository ytRepository,
                                          BannerModifyRepository modifyRepository,
                                          BannerUrlCheckService bannerUrlCheckService,
                                          UrlMonitoringNotifyService urlMonitoringNotifyService,
                                          InfoForUrlNotificationsGetter infoForUrlNotificationsGetter) {
        this.shardHelper = shardHelper;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytRepository = ytRepository;
        this.modifyRepository = modifyRepository;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.urlMonitoringNotifyService = urlMonitoringNotifyService;
        this.infoForUrlNotificationsGetter = infoForUrlNotificationsGetter;
    }

    @Override
    public void execute() {
        List<BannerUnreachableUrl> fetchedBanners = ytRepository.getAll();
        logger.info("fetched {} banners with unreachable url from YT: {}", fetchedBanners.size(),
                JsonUtils.toJson(fetchedBanners));

        Map<String, UrlCheckResult> urlCheckResultMap = getUrlCheckResultMap(fetchedBanners);

        List<BannerUnreachableUrl> unreachableBanners = getUnreachableBanners(fetchedBanners, urlCheckResultMap);
        logger.info("banners with unreachable url: {}", unreachableBanners);

        Predicate<BannerUnreachableUrl> isBannerIdNotDisable = getBannerIdNotDisablePredicate();
        Predicate<BannerUnreachableUrl> isBannerUrlNotDisable = getBannerUrlNotDisablePredicate();
        Predicate<BannerUnreachableUrl> isBannerErrorNotDisable = getBannerErrorNotDisablePredicate(urlCheckResultMap);

        List<BannerUnreachableUrl> filteredUnreachableBanners = getFilteredUnreachableBanners(unreachableBanners,
                isBannerIdNotDisable, isBannerUrlNotDisable, isBannerErrorNotDisable);

        boolean isStopBannersEnabled = ppcPropertiesSupport.get(BANNERS_UNREACHABLE_URL_STOP_BANNERS_ENABLED)
                .getOrDefault(false);

        List<BannerUnreachableUrl> stoppedBanners = getStoppedBanners(filteredUnreachableBanners, isStopBannersEnabled);
        List<StructureOfUnavailableBanners> structuredStoppedBanners = expandInfoOfUnavailableBanners(stoppedBanners);

        urlMonitoringNotifyService.notifyBannersStopped(structuredStoppedBanners, isStopBannersEnabled);

        List<BannerUnreachableUrl> notDisableBanners = getNotDisableBanners(unreachableBanners,
                filteredUnreachableBanners, stoppedBanners);
        List<StructureOfUnavailableBanners> structuredNotDisableBanners =
                expandInfoOfUnavailableBanners(notDisableBanners);

        urlMonitoringNotifyService.notifyBannersNotDisable(structuredNotDisableBanners, isBannerIdNotDisable,
                isBannerUrlNotDisable, isBannerErrorNotDisable);
    }

    Map<String, UrlCheckResult> getUrlCheckResultMap(List<BannerUnreachableUrl> fetchedBanners) {
        return StreamEx.of(fetchedBanners)
                .distinct(BannerUnreachableUrl::getUrl)
                .toMap(BannerUnreachableUrl::getUrl, banner -> bannerUrlCheckService.isUrlReachable(banner.getUrl()));
    }

    List<BannerUnreachableUrl> getUnreachableBanners(List<BannerUnreachableUrl> fetchedBanners,
                                                     Map<String, UrlCheckResult> urlCheckResultMap) {
        return filterList(fetchedBanners, banner -> !urlCheckResultMap.get(banner.getUrl()).getResult());
    }

    private <T> Set<T> getSetProperty(PpcPropertyName<Set<T>> propertyName) {
        return ppcPropertiesSupport.get(propertyName).getOrDefault(Collections.emptySet());
    }

    Predicate<BannerUnreachableUrl> getBannerIdNotDisablePredicate() {
        Set<Long> idsNotDisable = getSetProperty(BANNERS_UNREACHABLE_URL_IDS_NOT_DISABLE);
        return banner -> idsNotDisable.contains(banner.getId());
    }

    Predicate<BannerUnreachableUrl> getBannerUrlNotDisablePredicate() {
        Set<String> urlsNotDisable = getSetProperty(BANNERS_UNREACHABLE_URL_URLS_NOT_DISABLE);
        return banner -> StreamEx.of(urlsNotDisable)
                .anyMatch(url -> banner.getUrl().contains(url));
    }

    Predicate<BannerUnreachableUrl> getBannerErrorNotDisablePredicate(Map<String, UrlCheckResult> urlCheckResultMap) {
        Set<UrlCheckResult.Error> errorsNotDisable =
                StreamEx.of(getSetProperty(BANNERS_UNREACHABLE_URL_ERRORS_NOT_DISABLE))
                        .map(error -> EnumUtils.getEnum(UrlCheckResult.Error.class, error))
                        .nonNull()
                        .toSet();
        return banner -> errorsNotDisable.contains(urlCheckResultMap.get(banner.getUrl()).getError());
    }

    List<BannerUnreachableUrl> getFilteredUnreachableBanners(List<BannerUnreachableUrl> unreachableBanners,
                                                             Predicate<BannerUnreachableUrl> idsNotDisablePredicate,
                                                             Predicate<BannerUnreachableUrl> urlsNotDisablePredicate,
                                                             Predicate<BannerUnreachableUrl> errorsNotDisablePredicate) {
        return StreamEx.of(unreachableBanners)
                .remove(idsNotDisablePredicate)
                .remove(urlsNotDisablePredicate)
                .remove(errorsNotDisablePredicate)
                .toList();
    }

    List<BannerUnreachableUrl> getStoppedBanners(List<BannerUnreachableUrl> unreachableBanners,
                                                 boolean isStopBannersEnabled) {
        Map<Long, Set<String>> urlsByBannerId = getUrlsByBannerId(unreachableBanners);

        List<Long> bannerIds = List.copyOf(urlsByBannerId.keySet());
        Map<Integer, List<Long>> bannerIdsByShard = shardHelper.getBannerIdsByShard(bannerIds);

        Map<String, String> reasonByUrl = getReasonByUrl(unreachableBanners);

        List<BannerUnreachableUrl> stoppedBanners = new ArrayList<>();
        bannerIdsByShard.forEach((shard, ids) -> {
            List<BannerUnreachableUrl> currentStoppedBanners =
                    stopBanners(shard, ids, urlsByBannerId, reasonByUrl, isStopBannersEnabled);
            logger.info("{} banners stopped on the {} shard", currentStoppedBanners.size(), shard);
            stoppedBanners.addAll(currentStoppedBanners);
        });

        return stoppedBanners;
    }

    Map<Long, Set<String>> getUrlsByBannerId(List<BannerUnreachableUrl> banners) {
        return StreamEx.of(banners)
                .groupingBy(BannerUnreachableUrl::getId,
                        Collectors.mapping(BannerUnreachableUrl::getUrl, Collectors.toSet())
                );
    }

    Map<String, String> getReasonByUrl(List<BannerUnreachableUrl> banners) {
        return StreamEx.of(banners)
                .distinct(BannerUnreachableUrl::getUrl)
                .toMap(BannerUnreachableUrl::getUrl, BannerUnreachableUrl::getReason);
    }

    List<StructureOfUnavailableBanners> expandInfoOfUnavailableBanners(List<BannerUnreachableUrl> bannerUnreachableUrls) {
        Map<Long, List<BannerUnreachableUrl>> allBannersById = StreamEx.of(bannerUnreachableUrls)
                .groupingBy(BannerUnreachableUrl::getId);

        List<Long> allBannerIds = List.copyOf(allBannersById.keySet());

        Map<Integer, List<Long>> bannerIdsByShard = shardHelper.getBannerIdsByShard(allBannerIds);


        Map<Integer, Map<Long, List<BannerUnreachableUrl>>> bannersByShardAndId = EntryStream.of(bannerIdsByShard)
                .mapValues(ids -> StreamEx.of(ids)
                        .mapToEntry(Function.identity(), allBannersById::get)
                        .toMap()
                )
                .toMap();

        return EntryStream.of(bannersByShardAndId)
                .mapToValue((shard, bannersById) -> {
                    Set<Long> bannerIds = Set.copyOf(bannersById.keySet());
                    List<StructureOfBannerIds> additionalInfo =
                            infoForUrlNotificationsGetter.getAdditionalInfoByBannerIds(shard, bannerIds);
                    return StreamEx.of(additionalInfo)
                            .map(container -> new StructureOfUnavailableBanners(
                                    container.getCampaignId(), container.getCampaignName(),
                                    InfoForUrlNotificationsGetter.replaceBannerIdsWithUrls(container.getBannersByTemplateId(), bannersById))
                            );
                })
                .values().flatMap(Function.identity()).toList();
    }

    private Function<Long, ModelChanges<BannerWithInternalInfo>> getModelChangesCreator(boolean isStopBannersEnabled) {
        if (isStopBannersEnabled) {
            return id ->
                    new ModelChanges<>(id, BannerWithInternalInfo.class)
                            .process(false, BannerWithInternalInfo.STATUS_SHOW)
                            .process(StatusBsSynced.NO, BannerWithInternalInfo.STATUS_BS_SYNCED)
                            .process(true, BannerWithInternalInfo.IS_STOPPED_BY_URL_MONITORING);
        }
        return id -> new ModelChanges<>(id, BannerWithInternalInfo.class);
    }

    List<BannerUnreachableUrl> stopBanners(int shard, List<Long> bannerIds, Map<Long, Set<String>> urlsByBannerId,
                                           Map<String, String> reasonByUrl, boolean isStopBannersEnabled) {
        List<BannerUnreachableUrl> stoppedBanners = new ArrayList<>();

        ppcDslContextProvider.ppcTransactionResult(shard, configuration -> {
            DSLContext dslContext = DSL.using(configuration);

            List<ModelChanges<BannerWithInternalInfo>> modelChanges =
                    mapList(bannerIds, getModelChangesCreator(isStopBannersEnabled));

            return modifyRepository.updateByPredicate(
                    dslContext, new BannerRepositoryContainer(shard), modelChanges, BannerWithInternalInfo.class,
                    banner -> {
                        // Не трогаем остановленные баннеры
                        if (!banner.getStatusShow()) {
                            return false;
                        }
                        long id = banner.getId();
                        Set<String> unreachableUrls = urlsByBannerId.get(id);
                        // Недоступные урлы текущего баннера
                        List<String> currentUnreachableUrls =
                                StreamEx.of(banner.getTemplateVariables())
                                        .map(TemplateVariable::getInternalValue)
                                        .filter(unreachableUrls::contains)
                                        .toList();
                        StreamEx.of(currentUnreachableUrls)
                                .map(url -> createBannerUnreachableUrl(id, url, reasonByUrl.get(url)))
                                .forEach(stoppedBanners::add);
                        // Останавливаем баннер, если у него есть хотя бы один недоступный урл
                        return !currentUnreachableUrls.isEmpty();
                    }
            );
        });

        return stoppedBanners;
    }

    /**
     * Возвращает баннеры, которые не были остановлены из-за пропертей.
     * <p>
     * Логика метода примерно следующая: (баннеры с недоступными урлами) минус (баннеры, которые не попадают
     * не под какой-то из проперти) = (баннеры, попадающие под проперти). Но среди таких баннеров могут быть те, которые
     * содержат недоступные урлы, не попадающие не под какой проперти, поэтому
     * {@link UpdateBannersUnreachableUrlJob#stopBanners} их остановит. Из-за этого фильтруем по тем id, которые не
     * попали в список остановленных.
     * </p>
     */
    List<BannerUnreachableUrl> getNotDisableBanners(List<BannerUnreachableUrl> unreachableBanners,
                                                    List<BannerUnreachableUrl> filteredUnreachableBanners,
                                                    List<BannerUnreachableUrl> stoppedBanners) {
        List<BannerUnreachableUrl> bannersNotDisable = ListUtils.subtract(unreachableBanners,
                filteredUnreachableBanners);
        Set<Long> stoppedBannersId = listToSet(stoppedBanners, BannerUnreachableUrl::getId);
        return filterList(bannersNotDisable, banner -> !stoppedBannersId.contains(banner.getId()));
    }
}
