package ru.yandex.direct.jobs.internal;

import java.math.RoundingMode;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.href.BannerUrlCheckService;
import ru.yandex.direct.core.entity.internalads.model.MobileResourcesConfig;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.service.urlchecker.RedirectCheckResult;
import ru.yandex.direct.core.validation.constraints.MobileContentConstraints;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.core.entity.internalads.Constants.MOBILE_APP_TEMPLATE_IDS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Обновление ресурса "рейтинг" в template_variables внутренних баннеров для шаблонов мобильных приложений в
 * соответствии с ресурсом url, по которому берется новый рейтинг мобильного приложения. Так же для баннера
 * сбрасывается статус синхронизации с БК.
 * Количество баннеров мобильных приложений невелико (примерно 2000 баннеров), поэтому чанкование отсутствует.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 17),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_DMITANOSH,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 17),
        needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 8 * 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
class UpdateInternalMobileBannerRatingJob extends DirectShardedJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdateInternalMobileBannerRatingJob.class);

    private final BannerTypedRepository bannerTypedRepository;
    private final BannerUrlCheckService bannerUrlCheckService;
    private final MobileContentService mobileContentService;
    private final DslContextProvider ppcDslContextProvider;
    private final BannerModifyRepository modifyRepository;

    @Autowired
    UpdateInternalMobileBannerRatingJob(BannerTypedRepository bannerTypedRepository,
                                        BannerUrlCheckService bannerUrlCheckService,
                                        MobileContentService mobileContentService,
                                        DslContextProvider ppcDslContextProvider,
                                        BannerModifyRepository modifyRepository) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.mobileContentService = mobileContentService;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.modifyRepository = modifyRepository;
    }

    /**
     * Конструктор для тестов
     */
    UpdateInternalMobileBannerRatingJob(int shard,
                                        BannerTypedRepository bannerTypedRepository,
                                        BannerUrlCheckService bannerUrlCheckService,
                                        MobileContentService mobileContentService,
                                        DslContextProvider ppcDslContextProvider,
                                        BannerModifyRepository modifyRepository) {
        super(shard);
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerUrlCheckService = bannerUrlCheckService;
        this.mobileContentService = mobileContentService;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.modifyRepository = modifyRepository;
    }

    @Override
    public void execute() {
        Map<Long, InternalBanner> bannersByBannerIds = listToMap(
                bannerTypedRepository.getNoArchivedInternalBannersWithStatusShowYesByTemplateIds(
                        getShard(), MOBILE_APP_TEMPLATE_IDS.keySet()),
                InternalBanner::getId);

        if (bannersByBannerIds.isEmpty()) {
            logger.info("Banners for update were not found");
            return;
        }

        Map<Long, MobileTemplateVariablesHolder> templateVarsByBannerIds = processTemplateVariables(bannersByBannerIds);

        Map<Long, MobileContent> mobileContentByBannerIds = getMobileContents(templateVarsByBannerIds);

        // без логирования, так как ошибки при получении MobileContent логируются в другом месте
        if (mobileContentByBannerIds.isEmpty()) {
            return;
        }

        Map<Long, List<TemplateVariable>> templateVarsByBannerIdsForUpdate = getTemplateVariablesForUpdate(
                bannersByBannerIds, templateVarsByBannerIds, mobileContentByBannerIds);

        if (templateVarsByBannerIdsForUpdate.isEmpty()) {
            logger.info("Banners already updated");
            return;
        }

        int updated = storeNewTemplateVariables(templateVarsByBannerIdsForUpdate, bannersByBannerIds);

        if (updated != 0) {
            logger.info("{} banners updated", updated);
        }

        if (updated != templateVarsByBannerIdsForUpdate.size()) {
            logger.info("{} banners were changed during the update process",
                    templateVarsByBannerIdsForUpdate.size() - updated);
        }
    }

    static class MobileTemplateVariablesHolder {
        final String url;
        final String rating;

        MobileTemplateVariablesHolder(@Nullable String url, @Nullable String rating) {
            this.url = url;
            this.rating = rating;
        }

        public String getUrl() {
            return url;
        }

        /**
         * @param simplifyUrl определяет, нужно ли урезать параметры ссылки (после '?'), получаемой из ресурсов, для
         *                    уменьшения количества запросов получения редиректов
         */
        static MobileTemplateVariablesHolder fromTemplateVariables(List<TemplateVariable> templateVariables,
                                                                   MobileResourcesConfig resourceIds,
                                                                   boolean simplifyUrl) {

            Map<Long, TemplateVariable> templateVariableById = listToMap(templateVariables,
                    TemplateVariable::getTemplateResourceId);

            String url = templateVariableById.get(resourceIds.getUrlResourceId()).getInternalValue();
            String rating = templateVariableById.get(resourceIds.getRatingResourceId()).getInternalValue();

            if (rating == null && templateVariableById.containsKey(resourceIds.getUrlResourceId())) {
                rating = "0";
            }

            if (url != null && simplifyUrl) {
                int indexOfParams = url.indexOf('?');

                if (indexOfParams != -1) {
                    url = url.substring(0, indexOfParams);
                }
            }

            return new MobileTemplateVariablesHolder(url, rating);
        }
    }

    private Map<Long, MobileTemplateVariablesHolder> processTemplateVariables(
            Map<Long, InternalBanner> bannersByBannerIds) {

        return EntryStream.of(bannersByBannerIds)
                .mapValues(b -> MobileTemplateVariablesHolder.fromTemplateVariables(
                        b.getTemplateVariables(), MOBILE_APP_TEMPLATE_IDS.get(b.getTemplateId()), true)
                )
                .filterKeyValue((k, v) -> {
                    if (v.rating == null) {
                        Long ratingResourceId = MOBILE_APP_TEMPLATE_IDS.get(bannersByBannerIds.get(k).getTemplateId())
                                .getRatingResourceId();

                        logger.warn(
                                "Failed to find template_variable with the given resource_id = {} and bannerId = {}",
                                ratingResourceId, k
                        );
                    }

                    if (v.url == null) {
                        Long urlResourceId = MOBILE_APP_TEMPLATE_IDS.get(bannersByBannerIds.get(k).getTemplateId())
                                .getUrlResourceId();

                        logger.warn(
                                "Failed to find template_variable with the given resource_id = {} and bannerId = {}",
                                urlResourceId, k
                        );
                    }

                    return v.rating != null && v.url != null;
                })
                .toMap();
    }

    private Map<Long, MobileContent> getMobileContents(
            Map<Long, MobileTemplateVariablesHolder> templateVarsByBannerIds) {
        Map<String, List<Long>> bannerIdsByUrl = EntryStream.of(templateVarsByBannerIds)
                .mapValues(MobileTemplateVariablesHolder::getUrl)
                .invert()
                .grouping();

        Map<String, MobileAppStoreUrl> mobileAppStoreUrls = EntryStream.of(bannerIdsByUrl)
                .mapToValue((url, bannerIds) -> {
                    RedirectCheckResult redirectResult = bannerUrlCheckService.getRedirect(url);

                    if (!redirectResult.isSuccessful()) {
                        logger.warn("Failed to redirect by url \"{}\", bannerIds: {}", url, bannerIds);
                    }

                    return redirectResult;
                })
                .filterValues(RedirectCheckResult::isSuccessful)
                .mapToValue((url, r) -> {
                    Optional<MobileAppStoreUrl> parsedUrl = MobileAppStoreUrlParser.parse(r.getRedirectUrl());

                    if (parsedUrl.isEmpty() && MobileContentConstraints.isValidAppStoreDomain(r.getRedirectUrl())) {
                        // хотим логировать только урлы на стор, а урлы на сайт это норм, для них рейтинг не трогаем
                        logger.error("Failed to parse url \"{}\"; orig_url \"{}\"; bannerIds: {}",
                                r.getRedirectUrl(), url, bannerIdsByUrl.get(url));
                    }

                    return parsedUrl.orElse(null);
                })
                .nonNullValues()
                .toMap();

        Map<MobileAppStoreUrl, Optional<MobileContent>> urlsToMobileContent =
                mobileContentService.getMobileContentFromYt(getShard(), Set.copyOf(mobileAppStoreUrls.values()));

        return EntryStream.of(templateVarsByBannerIds)
                .mapValues(t -> mobileAppStoreUrls.get(t.url))
                .nonNullValues()
                .mapValues(urlsToMobileContent::get)
                .filterKeyValue((id, c) -> {
                    if (c.isEmpty()) {
                        logger.warn("Failed to get mobile content from url \"{}\" (bannerId = {})",
                                templateVarsByBannerIds.get(id).url, id);
                    }

                    return c.isPresent();
                })
                .mapValues(Optional::get)
                .toMap();
    }

    private Map<Long, List<TemplateVariable>> getTemplateVariablesForUpdate(
            Map<Long, InternalBanner> bannersByBannerIds,
            Map<Long, MobileTemplateVariablesHolder> templateVarsByBannerIds,
            Map<Long, MobileContent> mobileContentByBannerIds) {
        return EntryStream.of(mobileContentByBannerIds)
                .mapValues(MobileContent::getRating)
                .mapValues(r -> r.setScale(1, RoundingMode.HALF_EVEN).toPlainString())
                .filterKeyValue((bannerId, newRating) ->
                        !newRating.equals(templateVarsByBannerIds.get(bannerId).rating))
                .peekKeyValue((bannerId, newRating) ->
                        logger.info("For banner with id = {} actual rating is {}, but banner has {}",
                                bannerId, newRating, templateVarsByBannerIds.get(bannerId).rating))
                .mapToValue((id, newRating) -> {
                    InternalBanner banner = bannersByBannerIds.get(id);
                    Long ratingResourceId = MOBILE_APP_TEMPLATE_IDS.get(banner.getTemplateId()).getRatingResourceId();

                    TemplateVariable newRatingTemplateVar = new TemplateVariable()
                            .withTemplateResourceId(ratingResourceId)
                            .withInternalValue(newRating);

                    return StreamEx.of(banner.getTemplateVariables())
                            .map(t -> t.getTemplateResourceId().equals(ratingResourceId) ? newRatingTemplateVar : t)
                            .toList();
                })
                .toMap();
    }

    private int storeNewTemplateVariables(Map<Long, List<TemplateVariable>> templateVarsByBannerIdsForUpdate,
                                          Map<Long, InternalBanner> bannersByBannerIds) {
        if (templateVarsByBannerIdsForUpdate.isEmpty()) {
            return 0;
        }

        int shard = getShard();

        return ppcDslContextProvider.ppcTransactionResult(shard, configuration -> {
            DSLContext dslContext = DSL.using(configuration);

            List<ModelChanges<BannerWithInternalInfo>> modelChanges =
                    EntryStream.of(templateVarsByBannerIdsForUpdate)
                            .mapKeyValue((id, var) -> new ModelChanges<>(id, BannerWithInternalInfo.class)
                                    .process(var, BannerWithInternalInfo.TEMPLATE_VARIABLES)
                                    .process(StatusBsSynced.NO, BannerWithInternalInfo.STATUS_BS_SYNCED))
                            .toList();

            return modifyRepository.updateByPredicate(
                    dslContext, new BannerRepositoryContainer(shard), modelChanges, BannerWithInternalInfo.class,
                    b -> b.getTemplateVariables().equals(bannersByBannerIds.get(b.getId()).getTemplateVariables())
            );
        });
    }
}
