package ru.yandex.direct.jobs.internal;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.internalads.model.PagePlace;
import ru.yandex.direct.core.entity.internalads.repository.PagePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.PagePlaceYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.core.entity.page.service.PageService;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.db.PpcPropertyNames.PAGE_PLACE_LAST_UPDATE_UNIX_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Импортирует связь площадки (page_id) с местом (place_id)
 * <p>
 * Источником данных является YT таблица: {@link YtDbTables#PAGESLOTSEQUENCE}.
 * Сохраняем в ppcdict в таблицу: {@link ru.yandex.direct.dbschema.ppcdict.Tables#PAGE_PLACE}.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 300),
        needCheck = ProductionOnly.class,

        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_KOZOBRODOV,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 300),
        needCheck = NonProductionEnvironment.class,

        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60 * 2, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdatePagePlaceJob extends DirectJob {
    private static final Logger log = LoggerFactory.getLogger(UpdatePagePlaceJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PageService pageService;
    private final PagePlaceYtRepository ytRepository;
    private final PagePlaceRepository repository;

    @Autowired
    public UpdatePagePlaceJob(PpcPropertiesSupport ppcPropertiesSupport, PageService pageService,
                              PagePlaceYtRepository ytRepository, PagePlaceRepository repository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.pageService = pageService;
        this.ytRepository = ytRepository;
        this.repository = repository;
    }

    @Override
    public void execute() {
        long ytTableLastUpdateUnixTime = ytRepository.getLastUpdateUnixTime();
        PpcProperty<Long> lastUpdateUnixTimeProperty = ppcPropertiesSupport.get(PAGE_PLACE_LAST_UPDATE_UNIX_TIME);
        if (!needUpdate(ytTableLastUpdateUnixTime, lastUpdateUnixTimeProperty)) {
            log.info("page_place table is already updated");
            return;
        }

        var internalPages = pageService.getAllInternalAdPages();
        Set<Long> pageIdsToFilter = new HashSet<>(listToSet(internalPages, Page::getId));
        pageIdsToFilter.addAll(listToSet(internalPages, Page::getOrigPageId));
        List<PagePlace> ytFetchedRecords = ytRepository.getPagePlaces(pageIdsToFilter);
        log.info("fetched {} records from YT", ytFetchedRecords.size());
        checkState(!ytFetchedRecords.isEmpty(), "fetched records from YT can't be empty");

        List<PagePlace> mysqlFetchedRecords = repository.getAll();
        log.info("fetched {} records from MySQL", mysqlFetchedRecords.size());

        List<PagePlace> newRecordsToAdd = getNewRecordsToAdd(ytFetchedRecords, mysqlFetchedRecords);
        repository.add(newRecordsToAdd);
        log.info("added {} new records", newRecordsToAdd.size());

        List<PagePlace> oldRecordsToDelete = getOldRecordsToDelete(ytFetchedRecords, mysqlFetchedRecords);
        repository.delete(oldRecordsToDelete);
        log.info("deleted {} old records: {}", oldRecordsToDelete.size(), JsonUtils.toJson(oldRecordsToDelete));

        log.info("setting {} property value to '{}'", PAGE_PLACE_LAST_UPDATE_UNIX_TIME.getName(), ytTableLastUpdateUnixTime);
        lastUpdateUnixTimeProperty.set(ytTableLastUpdateUnixTime);
    }

    /**
     * Нужно ли обновлять MySQL таблицу {@link Tables#PAGE_PLACE}
     *
     * @param ytTableLastUpdateUnixTime  время последнего обновления данных в YT таблице {@link YtDbTables#PAGESLOTSEQUENCE}
     * @param lastUpdateUnixTimeProperty проперти, которая хранит время последнего обновления данных в MySQL таблице {@link Tables#PAGE_PLACE}
     * @return true/false
     */
    static boolean needUpdate(long ytTableLastUpdateUnixTime, PpcProperty<Long> lastUpdateUnixTimeProperty) {
        Long lastUpdateUnixTime = lastUpdateUnixTimeProperty.get();
        log.info("{} property value is '{}'. max_unix_time from YT is {}", PAGE_PLACE_LAST_UPDATE_UNIX_TIME.getName(),
                lastUpdateUnixTime, ytTableLastUpdateUnixTime);

        return lastUpdateUnixTime == null || lastUpdateUnixTime < ytTableLastUpdateUnixTime;
    }

    /**
     * Возвращает записи, которых нет в базе MySQL, но есть в YT'е
     */
    private static List<PagePlace> getNewRecordsToAdd(
            List<PagePlace> ytFetchedRecords, List<PagePlace> mysqlFetchedRecords) {
        return ListUtils.subtract(ytFetchedRecords, mysqlFetchedRecords);
    }

    /**
     * Возвращает записи, которые есть в базе MySQL, но нет в YT'е
     */
    private static List<PagePlace> getOldRecordsToDelete(
            List<PagePlace> ytFetchedRecords, List<PagePlace> mysqlFetchedRecords) {
        return ListUtils.subtract(mysqlFetchedRecords, ytFetchedRecords);
    }
}
