package ru.yandex.direct.jobs.internal;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlace;
import ru.yandex.direct.core.entity.internalads.repository.PlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.PlacesYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.db.PpcPropertyNames.PLACES_LAST_UPDATE_UNIX_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Обновляет данные плейсов внутренней рекламы
 * Источником данных является YT таблица экспортируемая из БК: {@link YtDbTables#PLACE}
 * Сохраняем в ppcdict в таблицу: {@link Tables#INTERNAL_AD_PLACES}
 * Джоба добавляет новые плейсы из YT'я, обновляет измененные плейсы и удаляет старые записи в MySQL, которых больше нет в YT'е
 * Для определения новых и измененных плейсов делается выгрузка всех записей из базы. Сейчас их кол-во примерно 1,5 тыс.
 * Существенного роста кол-во записей в будущем не предполагается
 * <p>
 * Мониторинг должен поднимать CRIT если в течение 3 часов job'а ни разу не завершился успешно.
 * Примерное время выполнения - 1 минута
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = TypicalEnvironment.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdatePlacesJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdatePlacesJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final PlacesYtRepository ytRepository;
    private final PlaceRepository placeRepository;

    @Autowired
    public UpdatePlacesJob(PpcPropertiesSupport ppcPropertiesSupport,
                           PlacesYtRepository ytRepository,
                           PlaceRepository placeRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytRepository = ytRepository;
        this.placeRepository = placeRepository;
    }

    @Override
    public void execute() {
        long ytTableLastUpdateUnixTime = ytRepository.getLastUpdateUnixTime();
        PpcProperty<Long> lastUpdateUnixTimeProperty = ppcPropertiesSupport.get(PLACES_LAST_UPDATE_UNIX_TIME);
        if (!needUpdate(ytTableLastUpdateUnixTime, lastUpdateUnixTimeProperty)) {
            logger.info("internal_ad_places table is already updated");
            return;
        }

        List<InternalAdPlace> ytFetchedPlaces = ytRepository.getAll();
        logger.info("fetched {} places from YT", ytFetchedPlaces.size());
        checkState(!ytFetchedPlaces.isEmpty(), "fetched places from YT can't be empty");

        List<InternalAdPlace> mysqlFetchedPlaces = placeRepository.getAll();
        logger.info("fetched {} places from MySQL", mysqlFetchedPlaces.size());

        List<InternalAdPlace> placesToAddOrUpdate = getPlacesToAddOrUpdate(ytFetchedPlaces, mysqlFetchedPlaces);
        placeRepository.addOrUpdate(placesToAddOrUpdate);
        logger.info("added or updated {} places", placesToAddOrUpdate.size());

        Set<Long> oldPlaceIdsToDelete = getOldPlaceIdsToDelete(ytFetchedPlaces, mysqlFetchedPlaces);
        placeRepository.delete(oldPlaceIdsToDelete);
        logger.info("deleted {} old places with ids: {}", oldPlaceIdsToDelete.size(), oldPlaceIdsToDelete);

        logger.info("setting {} property value to '{}'", PLACES_LAST_UPDATE_UNIX_TIME.getName(), ytTableLastUpdateUnixTime);
        lastUpdateUnixTimeProperty.set(ytTableLastUpdateUnixTime);
    }

    /**
     * Нужно ли обновлять MySQL таблицу {@link Tables#INTERNAL_AD_PLACES}
     *
     * @param ytTableLastUpdateUnixTime  время последнего обновления данных в YT таблице {@link YtDbTables#PLACE}
     * @param lastUpdateUnixTimeProperty проперти, которая хранит время последнего обновления данных
     *                                   в MySQL-таблице {@link Tables#INTERNAL_AD_PLACES}
     * @return true/false
     */
    static boolean needUpdate(long ytTableLastUpdateUnixTime, PpcProperty<Long> lastUpdateUnixTimeProperty) {
        Long lastUpdateUnixTime = lastUpdateUnixTimeProperty.get();
        logger.info("{} property value is '{}'. max_unix_time from YT is {}", PLACES_LAST_UPDATE_UNIX_TIME.getName(),
                lastUpdateUnixTime, ytTableLastUpdateUnixTime);

        return lastUpdateUnixTime == null || lastUpdateUnixTime < ytTableLastUpdateUnixTime;
    }

    /**
     * Возвращает плейсы, которые нужно добавить или обновить в базе MySQL
     */
    private static List<InternalAdPlace> getPlacesToAddOrUpdate(List<InternalAdPlace> ytFetchedPlaces,
                                                                List<InternalAdPlace> mysqlFetchedPlaces) {
        return ListUtils.subtract(ytFetchedPlaces, mysqlFetchedPlaces);
    }

    /**
     * Возвращает идентификаторы плейсов, которые есть в базе MySQL, но нет в YT'е
     */
    private static Set<Long> getOldPlaceIdsToDelete(List<InternalAdPlace> ytFetchedPlaces,
                                                    List<InternalAdPlace> mysqlFetchedPlaces) {
        Set<Long> ytFetchedPlaceIds = listToSet(ytFetchedPlaces, InternalAdPlace::getId);
        Set<Long> mysqlFetchedPlaceIds = listToSet(mysqlFetchedPlaces, InternalAdPlace::getId);

        return SetUtils.difference(mysqlFetchedPlaceIds, ytFetchedPlaceIds);
    }

}
