package ru.yandex.direct.jobs.internal;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.repository.TemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplatePlaceYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.db.PpcPropertyNames.TEMPLATE_PLACE_LAST_UPDATE_UNIX_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;

/**
 * Обновляет связи шаблона внутренней рекламы templateId с местом, где этот шаблон может размещаться placeId
 * Источником данных является YT таблица экспортируемая из БК: {@link YtDbTables#TEMPLATEPLACE}
 * Сохраняем в ppcdict в таблицу: {@link Tables#TEMPLATE_PLACE}
 * Джоба добавляет новые записи из YT'я и удаляет старые записи в MySQL, которых больше нет в YT'е
 * Операции UPDATE не может быть, т.к. в таблице {@link Tables#TEMPLATE_PLACE} хранится только составной первичный ключ: templateId и placeId
 * Для определения новых записей делается выгрузка всех записей из базы. Сейчас их кол-во примерно 450
 * Существенного роста кол-во записей в будущем не предполагается
 * <p>
 * Мониторинг должен поднимать CRIT если в течение 3 часов job'а ни разу не завершился успешно.
 * Примерное время выполнения - меньше 1 минуты
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_XY6ER,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateTemplatePlaceJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdateTemplatePlaceJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final TemplatePlaceYtRepository ytRepository;
    private final TemplatePlaceRepository templatePlaceRepository;

    @Autowired
    public UpdateTemplatePlaceJob(PpcPropertiesSupport ppcPropertiesSupport,
                                  TemplatePlaceYtRepository ytRepository,
                                  TemplatePlaceRepository templatePlaceRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytRepository = ytRepository;
        this.templatePlaceRepository = templatePlaceRepository;
    }

    @Override
    public void execute() {
        long ytTableLastUpdateUnixTime = ytRepository.getLastUpdateUnixTime();
        PpcProperty<Long> lastUpdateUnixTimeProperty = ppcPropertiesSupport.get(TEMPLATE_PLACE_LAST_UPDATE_UNIX_TIME);
        if (!needUpdate(ytTableLastUpdateUnixTime, lastUpdateUnixTimeProperty)) {
            logger.info("template_place table is already updated");
            return;
        }

        List<TemplatePlace> ytFetchedRecords = ytRepository.getAll();
        logger.info("fetched {} records from YT", ytFetchedRecords.size());
        checkState(!ytFetchedRecords.isEmpty(), "fetched records from YT can't be empty");

        List<TemplatePlace> mysqlFetchedRecords = templatePlaceRepository.getAll();
        logger.info("fetched {} records from MySQL", mysqlFetchedRecords.size());

        List<TemplatePlace> newRecordsToAdd = getNewRecordsToAdd(ytFetchedRecords, mysqlFetchedRecords);
        templatePlaceRepository.add(newRecordsToAdd);
        logger.info("added {} new records", newRecordsToAdd.size());

        List<TemplatePlace> oldRecordsToDelete = getOldRecordsToDelete(ytFetchedRecords, mysqlFetchedRecords);
        templatePlaceRepository.delete(oldRecordsToDelete);
        logger.info("deleted {} old records: {}", oldRecordsToDelete.size(), JsonUtils.toJson(oldRecordsToDelete));

        logger.info("setting {} property value to '{}'", TEMPLATE_PLACE_LAST_UPDATE_UNIX_TIME.getName(), ytTableLastUpdateUnixTime);
        lastUpdateUnixTimeProperty.set(ytTableLastUpdateUnixTime);
    }

    /**
     * Нужно ли обновлять MySQL таблицу {@link Tables#TEMPLATE_PLACE}
     *
     * @param ytTableLastUpdateUnixTime  время последнего обновления данных в YT таблице {@link YtDbTables#TEMPLATEPLACE}
     * @param lastUpdateUnixTimeProperty проперти, которая хранит время последнего обновления данных в MySQL таблице {@link Tables#TEMPLATE_PLACE}
     * @return true/false
     */
    static boolean needUpdate(long ytTableLastUpdateUnixTime, PpcProperty<Long> lastUpdateUnixTimeProperty) {
        Long lastUpdateUnixTime = lastUpdateUnixTimeProperty.get();
        logger.info("{} property value is '{}'. max_unix_time from YT is {}", TEMPLATE_PLACE_LAST_UPDATE_UNIX_TIME.getName(),
                lastUpdateUnixTime, ytTableLastUpdateUnixTime);

        return lastUpdateUnixTime == null || lastUpdateUnixTime < ytTableLastUpdateUnixTime;
    }

    /**
     * Возвращает записи, которых нет в базе MySQL, но есть в YT'е
     */
    private static List<TemplatePlace> getNewRecordsToAdd(List<TemplatePlace> ytFetchedRecords, List<TemplatePlace> mysqlFetchedRecords) {
        return ListUtils.subtract(ytFetchedRecords, mysqlFetchedRecords);
    }

    /**
     * Возвращает записи, которые есть в базе MySQL, но нет в YT'е
     */
    private static List<TemplatePlace> getOldRecordsToDelete(List<TemplatePlace> ytFetchedRecords, List<TemplatePlace> mysqlFetchedRecords) {
        return ListUtils.subtract(mysqlFetchedRecords, ytFetchedRecords);
    }

}
