package ru.yandex.direct.jobs.internal;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.repository.TemplateResourceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplateResourceYtRepository;
import ru.yandex.direct.core.entity.internalads.ytmodels.generated.YtDbTables;
import ru.yandex.direct.dbschema.ppcdict.Tables;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.db.PpcPropertyNames.TEMPLATE_RESOURCE_LAST_UPDATE_UNIX_TIME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_SPB_SERVER_SIDE_TEAM;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Обновляет данные ресурсов шаблона внутренней рекламы
 * Источником данных является YT таблица экспортируемая из БК: {@link YtDbTables#TEMPLATERESOURCE}
 * Сохраняем в ppcdict в таблицу: {@link Tables#TEMPLATE_RESOURCE}
 * Джоба добавляет новые ресурсы из YT'я, обновляет измененные ресурсы и удаляет старые записи в MySQL, которых больше нет в YT'е
 * Для определения новых и измененных ресурсов делается выгрузка всех записей из базы. Сейчас их кол-во примерно 4,5 тыс.
 * Существенного роста кол-во записей в будущем не предполагается
 * <p>
 * Мониторинг должен поднимать CRIT если в течение 3 часов job'а ни разу не завершился успешно.
 * Примерное время выполнения - 1 минута
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_XY6ER,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: джоба обновляет словарные данные, в среднем починка ждет неделю. Но иногда может срочно
        // понадобиться получить свежие данные - вероятность маленькая, но надо иметь ввиду. И в таком случае
        // приоритет будет PRIORITY_1
        tags = {DIRECT_PRIORITY_2, DIRECT_SPB_SERVER_SIDE_TEAM}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class UpdateTemplateResourceJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(UpdateTemplateResourceJob.class);

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final TemplateResourceYtRepository ytRepository;
    private final TemplateResourceRepository templateResourceRepository;

    @Autowired
    public UpdateTemplateResourceJob(PpcPropertiesSupport ppcPropertiesSupport,
                                     TemplateResourceYtRepository ytRepository,
                                     TemplateResourceRepository templateResourceRepository) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.ytRepository = ytRepository;
        this.templateResourceRepository = templateResourceRepository;
    }

    @Override
    public void execute() {
        long ytTableLastUpdateUnixTime = ytRepository.getLastUpdateUnixTime();
        PpcProperty<Long> lastUpdateUnixTimeProperty = ppcPropertiesSupport.get(TEMPLATE_RESOURCE_LAST_UPDATE_UNIX_TIME);
        if (!needUpdate(ytTableLastUpdateUnixTime, lastUpdateUnixTimeProperty)) {
            logger.info("template_resource table is already updated");
            return;
        }

        List<TemplateResource> ytFetchedResources = ytRepository.getAll();
        logger.info("fetched {} resources from YT", ytFetchedResources.size());
        checkState(!ytFetchedResources.isEmpty(), "fetched resources from YT can't be empty");

        List<TemplateResource> mysqlFetchedResources = templateResourceRepository.getAll();
        logger.info("fetched {} resources from MySQL", mysqlFetchedResources.size());

        List<TemplateResource> resourceToAddOrUpdate = getResourcesToAddOrUpdate(ytFetchedResources, mysqlFetchedResources);
        templateResourceRepository.addOrUpdate(resourceToAddOrUpdate);
        logger.info("added or updated {} resources", resourceToAddOrUpdate.size());

        Set<Long> oldResourceIdsToDelete = getOldResourceIdsToDelete(ytFetchedResources, mysqlFetchedResources);
        templateResourceRepository.delete(oldResourceIdsToDelete);
        logger.info("deleted {} old resources with ids: {}", oldResourceIdsToDelete.size(), oldResourceIdsToDelete);

        logger.info("setting {} property value to '{}'", TEMPLATE_RESOURCE_LAST_UPDATE_UNIX_TIME.getName(), ytTableLastUpdateUnixTime);
        lastUpdateUnixTimeProperty.set(ytTableLastUpdateUnixTime);
    }

    /**
     * Нужно ли обновлять MySQL таблицу {@link Tables#TEMPLATE_RESOURCE}
     *
     * @param ytTableLastUpdateUnixTime  время последнего обновления данных в YT таблице {@link YtDbTables#TEMPLATERESOURCE}
     * @param lastUpdateUnixTimeProperty проперти, которая хранит время последнего обновления данных в MySQL таблице {@link Tables#TEMPLATE_RESOURCE}
     * @return true/false
     */
    static boolean needUpdate(long ytTableLastUpdateUnixTime, PpcProperty<Long> lastUpdateUnixTimeProperty) {
        Long lastUpdateUnixTime = lastUpdateUnixTimeProperty.get();
        logger.info("{} property value is '{}'. max_unix_time from YT is {}", TEMPLATE_RESOURCE_LAST_UPDATE_UNIX_TIME.getName(),
                lastUpdateUnixTime, ytTableLastUpdateUnixTime);

        return lastUpdateUnixTime == null || lastUpdateUnixTime < ytTableLastUpdateUnixTime;
    }

    /**
     * Возвращает ресурсы, которые нужно добавить или обновить в базе MySQL
     */
    private static List<TemplateResource> getResourcesToAddOrUpdate(List<TemplateResource> ytFetchedResources,
                                                                    List<TemplateResource> mysqlFetchedResources) {
        return ListUtils.subtract(ytFetchedResources, mysqlFetchedResources);
    }

    /**
     * Возвращает идентификаторы ресурсов, которые есть в базе MySQL, но нет в YT'е
     */
    private static Set<Long> getOldResourceIdsToDelete(List<TemplateResource> ytFetchedResources, List<TemplateResource> mysqlFetchedResources) {
        Set<Long> ytFetchedResourceIds = listToSet(ytFetchedResources, TemplateResource::getId);
        Set<Long> mysqlFetchedResourceIds = listToSet(mysqlFetchedResources, TemplateResource::getId);

        return SetUtils.difference(mysqlFetchedResourceIds, ytFetchedResourceIds);
    }

}
