package ru.yandex.direct.jobs.internal;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.internalads.model.BannerUnreachableUrl;
import ru.yandex.direct.jobs.internal.model.NotificationsUrlMonitoringMailMessage;
import ru.yandex.direct.jobs.internal.model.StructureOfBannerIds;
import ru.yandex.direct.jobs.internal.model.StructureOfBanners;
import ru.yandex.direct.jobs.internal.model.StructureOfUnavailableBanners;
import ru.yandex.direct.mail.MailSender;

@Service
@ParametersAreNonnullByDefault
public class UrlMonitoringNotifyService {
    private static final Logger logger = LoggerFactory.getLogger(UrlMonitoringNotifyService.class);

    private static final Function<BannerUnreachableUrl, String> BANNER_STOPPED_MESSAGE_MAPPER = banner ->
            String.format("bannerId: %d, url: %s, reason: %s", banner.getId(), banner.getUrl(), banner.getReason());

    private final MailSender mailSender;

    @Autowired
    UrlMonitoringNotifyService(MailSender mailSender) {
        this.mailSender = mailSender;
    }

    void notifyBannersStopped(List<StructureOfUnavailableBanners> stoppedBanners, boolean isStopBannersEnabled) {
        String subjectMessage = isStopBannersEnabled
                ? "Остановлены баннеры с недоступными урлами"
                : "Остановлены баннеры с недоступными урлами (на самом деле нет)";
        notifyUnavailableBanners(stoppedBanners, BANNER_STOPPED_MESSAGE_MAPPER, subjectMessage);
    }

    void notifyBannersNotDisable(List<StructureOfUnavailableBanners> notDisableBanners,
                                 Predicate<BannerUnreachableUrl> idsNotDisablePredicate,
                                 Predicate<BannerUnreachableUrl> urlsNotDisablePredicate,
                                 Predicate<BannerUnreachableUrl> errorsNotDisablePredicate) {
        notifyUnavailableBanners(
                notDisableBanners,
                getNotDisableBannersMessageMapper(
                        idsNotDisablePredicate, urlsNotDisablePredicate, errorsNotDisablePredicate
                ),
                "Отфильтрованные баннеры:"
        );
    }

    void notifyBannersEnabled(List<StructureOfBannerIds> structures) {
        if (structures.isEmpty()) {
            return;
        }

        String messageText = StreamEx.of(structures)
                .sorted(StructureOfBanners.comparingByCampaignId())
                .map(structure ->
                        String.format(
                                "Кампания №%d \"%s\":\n%s",
                                structure.getCampaignId(), structure.getCampaignName(),
                                EntryStream.of(structure.getBannersByTemplateId())
                                        .sorted(Map.Entry.comparingByKey())
                                        .mapKeys(templateId -> "Шаблон " + templateId)
                                        .mapValues(ids -> StreamEx.of(ids).sorted().joining(", "))
                                        .join(": ")
                                        .joining("\n")
                        )
                )
                .joining("\n\n");

        notifyMessage(messageText, "Включены баннеры, остановленные урл-мониторингом");
    }

    private Function<BannerUnreachableUrl, String> getNotDisableBannersMessageMapper(
            Predicate<BannerUnreachableUrl> idsNotDisablePredicate,
            Predicate<BannerUnreachableUrl> urlsNotDisablePredicate,
            Predicate<BannerUnreachableUrl> errorsNotDisablePredicate) {
        return banner -> BANNER_STOPPED_MESSAGE_MAPPER.apply(banner) +
                " - idsNotDisable=" +
                idsNotDisablePredicate.test(banner) +
                ", urlsNotDisable=" +
                urlsNotDisablePredicate.test(banner) +
                ", errorsNotDisable=" +
                errorsNotDisablePredicate.test(banner);
    }

    private void notifyUnavailableBanners(
            List<StructureOfUnavailableBanners> structures, Function<BannerUnreachableUrl, String> bannerMessageMapper,
            String subjectMessage
    ) {
        if (structures.isEmpty()) {
            return;
        }

        String textMessage = StreamEx.of(structures)
                .sorted(StructureOfBanners.comparingByCampaignId())
                .map(structure -> String.format(
                        "Кампания №%d \"%s\":\n%s",
                        structure.getCampaignId(), structure.getCampaignName(),
                        EntryStream.of(structure.getBannersByTemplateId())
                                .sorted(Map.Entry.comparingByKey())
                                .mapKeyValue((templateId, banners) -> String.format(
                                        "Шаблон %d:\n%s",
                                        templateId,
                                        StreamEx.of(banners)
                                                .sorted(Comparator.comparing(BannerUnreachableUrl::getId))
                                                .map(bannerMessageMapper)
                                                .joining("\n")
                                ))
                                .joining("\n")
                ))
                .joining("\n\n");

        notifyMessage(textMessage, subjectMessage);
    }

    private void notifyMessage(String messageText, String subjectMessage) {
        var message = new NotificationsUrlMonitoringMailMessage(subjectMessage, messageText);

        mailSender.send(message);

        logger.info("mail text:\n{}", message);
    }
}
