package ru.yandex.direct.jobs.internal.utils

import one.util.streamex.StreamEx
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.internalads.model.BannerUnreachableUrl
import ru.yandex.direct.jobs.internal.model.StructureOfBannerIds
import java.util.function.Function

@Component
class InfoForUrlNotificationsGetter(
    private val campaignRepository: CampaignRepository,
    private val bannerTypedRepository: BannerTypedRepository,
) {
    fun getAdditionalInfoByBannerIds(
        shard: Int,
        bannerIds: Set<Long>
    ): List<StructureOfBannerIds> {
        val banners = bannerTypedRepository.getInternalBannersByIds(shard, bannerIds)
        return getAdditionalInfoByInternalBanners(shard, banners)
    }

    fun getAdditionalInfoByInternalBanners(
        shard: Int,
        banners: List<InternalBanner>
    ): List<StructureOfBannerIds> {
        val campaignNames = getCampaignNamesFromItsBanners(shard, banners)
        return structureBanners(banners, campaignNames)
    }

    fun getCampaignNamesFromItsBanners(shard: Int, banners: List<InternalBanner>): Map<Long, String> {
        val campaignIds = banners.map { it.campaignId }

        val campaigns = campaignRepository.getCampaignsSimple(shard, campaignIds)
        return campaigns.mapValues { (_, campaign) -> campaign.name }
    }

    companion object {
        @JvmStatic
        fun structureBanners(
            banners: List<InternalBanner>,
            campaignNames: Map<Long, String>,
        ): List<StructureOfBannerIds> {
            return banners
                .groupBy { it.campaignId }
                .mapValues { (_, value) ->
                    value.groupBy { it.templateId }
                        .mapValues { it.value.map(InternalBanner::getId) }
                }
                .map { (campaignId, bannersByTemplateId) ->
                    StructureOfBannerIds(
                        campaignId, campaignNames[campaignId]!!, bannersByTemplateId
                    )
                }
        }

        @JvmStatic
        fun replaceBannerIdsWithUrls(
            bannerIdsByTemplateId: Map<Long, List<Long>>, bannerUrlsById: Map<Long, List<BannerUnreachableUrl>>
        ): Map<Long, List<BannerUnreachableUrl>> {
            return bannerIdsByTemplateId
                .mapValues { (_, banners) ->
                    StreamEx.of(banners).map(bannerUrlsById::get).toFlatList(Function.identity())
                }
                .toMap()
        }
    }
}
