package ru.yandex.direct.jobs.inventoribookedtargettags;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion.SQLv1;

@ParametersAreNonnullByDefault
@JugglerCheck(
        ttl = @JugglerCheck.Duration(hours = 72),
        tags = {DIRECT_PRIORITY_2},
        needCheck = ProductionOnly.class,
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_ALEX_KULAKOV,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 0 0/2 * * ?", needSchedule = TypicalEnvironment.class)
public class TargetTagsFromYTToMySQLJob extends DirectJob {

    private static final Logger logger = LoggerFactory.getLogger(TargetTagsFromYTToMySQLJob.class);

    private static final String REQUEST_PATH = "classpath:///inventoribookedtargettags/targettagsfromyttomysqljob.yql";
    private static final YtCluster CLUSTER = YtCluster.HAHN;

    private final String query;
    private final YtProvider ytProvider;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private static final int MAX_SIZE_OF_TEXT_IN_MYSQL = 65535;



    public TargetTagsFromYTToMySQLJob(YtProvider ytProvider, PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.query = LiveResourceFactory.get(REQUEST_PATH).getContent();
    }

    ResultSetFuture runQuery() throws SQLException {
        var ytOperator = this.ytProvider.getOperator(YtCluster.HAHN);

        logger.info("Running YQL.");
        var startingQuery = ytOperator.yqlQueryBegin(new YqlQuery(this.query));
        return (ResultSetFuture) checkOperation(startingQuery.getOperationId());
    }

    Future<?> checkOperation(String operationId) throws SQLException {
        YqlConnection connection = (YqlConnection) ytProvider.getYql(CLUSTER, SQLv1).getConnection();
        return connection.restoreResultSetFuture(operationId, "");
    }

    public String makeNewValue(ResultSetFuture resultSetFuture) throws SQLException, ExecutionException,
            InterruptedException {
        ResultSet rs = resultSetFuture.get();
        logger.info("YQL completed.");
        StringBuilder stringBuilder = new StringBuilder();
        while (rs.next()) {
            String targetTag = rs.getString("target_tag");
            stringBuilder.append(targetTag );
            stringBuilder.append(", ");
        }
        stringBuilder.delete(stringBuilder.length() - 2, stringBuilder.length() - 1);
        return stringBuilder.toString();
    }

    @Override
    public void execute() {
        try {
            var resultSetFuture = runQuery();
            String newValue = makeNewValue(resultSetFuture);
            if (newValue.length() > MAX_SIZE_OF_TEXT_IN_MYSQL) {
                logger.error("Длинна нового значения слишком велика, перезаписи не произошло");
                return;
            }
            ppcPropertiesSupport.set("inventori_target_tags", newValue);
            logger.info("Значение успешно перезаписалось");
        } catch (SQLException | ExecutionException | InterruptedException e) {
            logger.error("Произошла ошибка, значение не изменилось");
        }
    }
}
