package ru.yandex.direct.jobs.logs;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.env.NonDevelopmentEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.jobs.bannersystem.logs.configuration.BsExportLogsLogbrokerConsumerProperties;
import ru.yandex.direct.jobs.logs.configuration.YtBsExportLogParametersSource;
import ru.yandex.direct.jobs.logs.configuration.YtLogParameter;
import ru.yandex.direct.jobs.logs.deserializer.BsExportLogDeserializer;
import ru.yandex.direct.jobs.logs.model.BsExportLogRow;
import ru.yandex.direct.jobs.logs.serializer.BsExportLogWireSerializer;
import ru.yandex.direct.jobs.logs.serializer.YtWriterException;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.MonotonicClock;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig;
import ru.yandex.direct.ytwrapper.model.LockWrapper;
import ru.yandex.direct.ytwrapper.model.OperationWrapper;
import ru.yandex.direct.ytwrapper.model.TableMerger;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.ConsumerReadResponse;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.TableWriter;
import ru.yandex.yt.ytclient.proxy.request.AlterTable;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.CreateNode;
import ru.yandex.yt.ytclient.proxy.request.ExistsNode;
import ru.yandex.yt.ytclient.proxy.request.GetNode;
import ru.yandex.yt.ytclient.proxy.request.ObjectType;
import ru.yandex.yt.ytclient.proxy.request.SetNode;
import ru.yandex.yt.ytclient.proxy.request.StartTransaction;
import ru.yandex.yt.ytclient.proxy.request.WriteTable;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM;
import static ru.yandex.direct.jobs.logs.deserializer.BsExportLogDeserializer.TYPE_REFERENCE;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1_NOT_READY;
import static ru.yandex.direct.juggler.check.model.CheckTag.GROUP_INTERNAL_SYSTEMS;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.CHAT_INTERNAL_SYSTEMS_MONITORING;
import static ru.yandex.direct.utils.DateTimeUtils.fromEpochMillis;
import static ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToInstant;
import static ru.yandex.direct.ytwrapper.YtUtils.COMPRESSED_DATA_SIZE;
import static ru.yandex.direct.ytwrapper.YtUtils.COMPRESSION_CODEC_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.ERASURE_CODEC_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.EXPIRATION_TIME_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.SCHEMA_ATTR;
import static ru.yandex.direct.ytwrapper.YtUtils.SORTED_ATTR;
import static ru.yandex.direct.ytwrapper.model.YtDynamicOperator.getWithTimeout;

@HourglassDaemon(maxExecuteIterations = 30)
@Hourglass(needSchedule = NonDevelopmentEnvironment.class, periodInSeconds = 30)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        notifications = @OnChangeNotification(recipient = CHAT_INTERNAL_SYSTEMS_MONITORING,
                method = TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS}
)
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 3),
        needCheck = TestingOnly.class,
        tags = {DIRECT_PRIORITY_1_NOT_READY, GROUP_INTERNAL_SYSTEMS}
)
@ParameterizedBy(parametersSource = YtBsExportLogParametersSource.class)
@ParametersAreNonnullByDefault
public class YtBsExportLog extends DirectParameterizedJob<YtLogParameter> {
    private static final Logger logger = LoggerFactory.getLogger(YtBsExportLog.class);

    private static final int LOGBROKER_PARTITIONS_COUNT = 26;

    private static final String LOGS_FOLDER = "//home/direct/logs/bsexport_log";
    /**
     * Какого размера чанки хотим в таблицах с логом (чтобы не потребить всю квоту аккаунта на чанки в штуках)
     */
    private static final long DESIRED_CHUNK_SIZE = 1024L * 1024L * 1024L;
    /**
     * На сколько можем превышать фактическое количество чанков над расчитанным по {@link #DESIRED_CHUNK_SIZE} без мержа
     * <p>
     * Нужно, чтобы не делать merge-операцию после каждого записанного чанка.
     * Так получается, если не хватает данных или таймаута для записи {@link #DESIRED_CHUNK_SIZE) байт
     */
    private static final long CHUNKS_GAP = 100;

    private static final Duration COMMIT_TX_INTERVAL = Duration.ofMinutes(10);
    private static final Duration WRITE_TIMEOUT = Duration.ofMinutes(8);
    // TODO: посмотреть в бою (хан, арнольд) сколько времени занимает мерж и уменьшить
    private static final Duration MERGE_TIMEOUT = Duration.ofMinutes(20);
    private static final String MERGE_OPERATIONS_POOL = "direct-process-logbroker-log";
    private static final Duration LOCK_WAIT_TIMEOUT = Duration.ofHours(1);

    private static final DateTimeFormatter DATE_TIME_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private final YtProvider ytProvider;
    private final YtBsExportLogParametersSource parametersSource;

    private final LogbrokerClientFactoryFacade logbrokerClientFactory;
    private final JsonFactory jsonFactory;
    private final BsExportLogWireSerializer serializer;
    private final Map<Object, Runnable> interruptable;
    private final List<Long> cookies;
    private final List<Integer> groups;
    private final Map<LocalDate, TableOperator> operators;
    private final MonotonicClock clock;
    private final int numOfPpcShards;

    private SyncConsumer logbroker;
    private YtDynamicOperator yt;
    private YtCluster logsYtCluster;
    private ApiServiceTransaction writeTx;
    // таймстемп есть в самой транзакции, но его не очень корректно сравнивать с локальными часами
    private MonotonicTime lastTxStart;
    private MonotonicTime lastTxPing;

    private volatile boolean shutdown;
    private boolean firstRun;

    @Autowired
    public YtBsExportLog(YtProvider ytProvider, YtDynamicConfig ytDynamicConfig,
                         YtBsExportLogParametersSource parametersSource,
                         TvmIntegration tvmIntegration,
                         @Value("${db_shards:-1}") int numOfPpcShards
    ) {
        this.ytProvider = ytProvider;
        this.parametersSource = parametersSource;
        checkState(ytDynamicConfig.streamingWriteTimeout().compareTo(WRITE_TIMEOUT) > 0,
                "streaming-write-timeout should be greater than WRITE_TIMEOUT");
        checkState(ytDynamicConfig.streamingReadTimeout().compareTo(WRITE_TIMEOUT) > 0,
                "streaming-read-timeout should be greater than WRITE_TIMEOUT");


        this.numOfPpcShards = numOfPpcShards;

        Supplier<Credentials> credentialsSupplier = () -> {
            String serviceTicket = tvmIntegration.getTicket(TvmService.LOGBROKER_PRODUCTION);
            return Credentials.tvm(serviceTicket);
        };
        this.logbrokerClientFactory = new LogbrokerClientFactoryFacade(credentialsSupplier);

        groups = new ArrayList<>();
        cookies = new ArrayList<>();
        interruptable = new ConcurrentHashMap<>();
        operators = new HashMap<>();

        serializer = new BsExportLogWireSerializer();
        jsonFactory = new ObjectMapper()
                .registerModule(new SimpleModule()
                        .addDeserializer(BsExportLogRow.class, new BsExportLogDeserializer(true)))
                .getFactory();

        clock = NanoTimeClock.CLOCK;

        firstRun = true;
    }

    private void startWriteTx() {
        checkState(writeTx == null, "Previous transaction was not properly closed");

        StartTransaction startTransaction = StartTransaction.master()
                .setTransactionTimeout(WRITE_TIMEOUT)
                .setPingPeriod(null)
                .setPing(false);
        writeTx = yt.runRpcCommandWithTimeout(yt.getYtClient()::startTransaction, startTransaction);
        lastTxPing = lastTxStart = clock.getTime();
        logger.info("Started write transaction {}", writeTx);
    }

    private void pingWriteTx() {
        checkState(writeTx != null, "No opened transaction");

        MonotonicTime now = clock.getTime();
        if (lastTxPing != null && now.minus(Duration.ofSeconds(5)).isBefore(lastTxPing)) {
            logger.trace("Write transaction {} was recently pinged", writeTx);
            return;
        }

        logger.debug("Ping write transaction {}", writeTx);
        yt.runRpcCommandWithTimeout(writeTx::ping);
        lastTxPing = now;
    }

    private void abortWriteTx() {
        checkState(writeTx != null, "No opened transaction");
        logger.info("Abort write transaction {}", writeTx);
        yt.runRpcCommandWithTimeout(writeTx::abort);
        writeTx = null;
    }

    private void commitWriteTx() {
        checkState(writeTx != null, "No opened transaction");
        logger.info("Commit write transaction {}", writeTx);
        yt.runRpcCommandWithTimeout(writeTx::commit);
        writeTx = null;
        lastTxPing = lastTxStart = null;
    }

    @Override
    public void initialize(TaskParametersMap taskParametersMap) {
        super.initialize(taskParametersMap);

        var params = parametersSource.convertStringToParam(getParam());
        logsYtCluster = params.getYtCluster();
        yt = ytProvider.getDynamicOperator(logsYtCluster);


        int parts = YtLogParameter.WORKERS_NUM;
        int part = params.getWorkerNum();
        // Почему группы тут, а не сразу в param? Чтобы через ppc_properties можно было уменьшать число воркеров
        for (int i = 1; i <= LOGBROKER_PARTITIONS_COUNT; i++) {
            if (i % parts == part) {
                groups.add(i);
            }
        }

        BsExportLogsLogbrokerConsumerProperties logbrokerConsumerProperties = BsExportLogsLogbrokerConsumerProperties
                .toStaticTablesFromProd()
                .setConsumer(params.getConsumer())
                .setGroups(groups).build();
        logbroker = logbrokerClientFactory.createConsumerSupplier(logbrokerConsumerProperties).get();
    }

    private void logSettingsOnFirstIteration() {
        if (firstRun) {
            // если писать логи из конструктора или initialize() то их потом фиг найдешь, т.к. там нет трейс-метода
            if (numOfPpcShards > LOGBROKER_PARTITIONS_COUNT) {
                // если шардов становится больше, то нужно добавить партиций в топик и поднять тут константу
                logger.warn("You may need to increase LOGBROKER_PARTITIONS_COUNT constant");
            }
            logger.info("Processing logbroker groups: {}", groups);
            firstRun = false;
        }
    }

    @Override
    public void execute() {
        checkState(logbroker != null, "Logbroker client is not initialized or was closed");
        logSettingsOnFirstIteration();

        logger.info("Start iteration");
        writeIteration();

        List<TableOperator> mergeNeeded = new ArrayList<>();
        if (!shutdown) {
            logger.debug("Check chunks count in operated tables");
            StreamEx.ofValues(operators)
                    .filter(o -> o.getAvailableChunksCount() < 0)
                    .forEach(mergeNeeded::add);
        }

        logger.info("Iteration finished, close writers, commit tx and cookies");
        closeWritersAndCommit();

        if (!mergeNeeded.isEmpty()) {
            for (TableOperator operator : mergeNeeded) {
                logger.info("Too small chunks in {}. Merge table", operator.getTablePath());
                operator.mergeTable();
            }
        }
    }

    private boolean needRestartWriting() {
        if (shutdown) {
            logger.info("Restart writing due to graceful shutdown requested");
            return true;
        }

        MonotonicTime now = clock.getTime();
        if (lastTxStart.plus(COMMIT_TX_INTERVAL).isBefore(now)) {
            logger.info("Restart writing by transaction duration");
            return true;
        }

        for (TableOperator operator : operators.values()) {
            if (operator.getBytesWrittenApprox() * 0.1 > DESIRED_CHUNK_SIZE) {
                // 0.1 — это примерное значение compression_ratio для табличек с логами экспорта
                logger.info("Restart writing due to achieved estimated chunk size in {}", operator.getTablePath());
                return true;
            }
            MonotonicTime firstWriteTime = operator.getFirstWriteTime();
            if (firstWriteTime != null && now.minus(WRITE_TIMEOUT).isAfter(firstWriteTime)) {
                /*
                 * пытаемся избежать writeTimeout и readTimeout в DefaultRpcBusClient
                 * writeTimeout сейчас не продлевается совсем
                 * readTimeout продлевается при отправке очередного чанка данных
                 * если починить writeTimeout — можно запоминать время последней (а не первой) записи и сравнивать с ним
                 */
                logger.info("Restart writing to prevent streaming timeout for {}", operator.getTablePath());
                return true;
            }
        }

        return false;
    }

    private void writeIteration() {
        /*
         * В теории под одной кукой могут приехать сообщения из разных суток.
         * Чтобы упросить код: немного жертвуем атомарностью записи (единая транзакция YT для всех подневых таблиц)
         * и числом чанков (сначала пишем, и только потом проверяем "не много ли чанков"
         */

        startWriteTx();

        while (!needRestartWriting()) {
            pingWriteTx();

            ConsumerReadResponse logbrokerResponse;
            try (TraceProfile ignored = Trace.current().profile("logbroker:waitForData")) {
                logbrokerResponse = logbroker.read();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.warn("Logbroker reading interrupted, abort tx", e);
                abortWriteTx();
                throw new InterruptedRuntimeException(e);
            }
            pingWriteTx();

            if (logbrokerResponse == null || logbrokerResponse.getBatches().isEmpty()) {
                logger.debug("Empty response from logbroker");
                continue;
            }

            for (MessageBatch batch : logbrokerResponse.getBatches()) {
                logger.trace("process new batch from logbroker");
                for (MessageData messageData : batch.getMessageData()) {
                    logger.trace("process new message in batch");
                    LocalDateTime writeTime = fromEpochMillis(messageData.getMessageMeta().getWriteTimeMs());

                    byte[] decompressedData = messageData.getDecompressedData();

                    List<BsExportLogRow> rows;
                    try (TraceProfile ignored = Trace.current().profile("parse");
                         JsonParser parser = jsonFactory.createParser(decompressedData)
                    ) {
                        rows = parser.readValueAs(TYPE_REFERENCE);
                        checkState(!rows.isEmpty(), "Empty rows array");
                    } catch (IOException | IllegalStateException e) {
                        logger.error("Failed to parse data from logbroker", e);
                        logger.info("Partition: {}, offset: {}, message: {}...",
                                batch.getPartition(), messageData.getOffset(),
                                new String(decompressedData, 0, 2048, StandardCharsets.UTF_8));
                        //TODO DIRECT-135514: писать нераспаршенные данные в отдельную табличку
                        throw new UnsupportedOperationException();
                    }

                    String logTime = writeTime.format(DATE_TIME_FORMAT);
                    rows.forEach(row -> row.setLogTime(logTime));

                    TableOperator operator = operators.computeIfAbsent(writeTime.toLocalDate(), TableOperator::new);
                    operator.createTableIfNotExists();
                    operator.alterTableIfNeeded();

                    if (cookies.isEmpty()                               // ничего не собираемся коммитить
                            && !operator.hasWriter()                    // еще не начали писать в табличку
                            && operators.size() == 1                    // таблица пока только одна
                            && operator.getAvailableChunksCount() < 0   // и её пора бы смержить
                    ) {
                        abortWriteTx(); // иначе протухнет по таймауту

                        logger.info("Too small chunks in {}. Merge table", operator.getTablePath());

                        boolean success = operator.mergeTable();
                        if (!success) {
                            throw new RuntimeException("Table " + operator.getTablePath()
                                    + " has too much small chunks");
                        }

                        startWriteTx();
                    }

                    // примерно считаем, что обратно в YT пишем столько же, сколько прочитали из логброкера
                    operator.write(rows, decompressedData.length);
                    pingWriteTx();
                }
            }
            cookies.add(logbrokerResponse.getCookie());
        }
    }

    private void closeWritersAndCommit() {
        List<RuntimeException> suppressed = new ArrayList<>();
        List<String> tables = new ArrayList<>();

        for (TableOperator operator : operators.values()) {
            try {
                tables.add(operator.getTablePath());
                operator.closeWriter();
            } catch (RuntimeException e) {
                suppressed.add(e);
            }
        }
        operators.clear();

        if (!suppressed.isEmpty()) {
            var e = new YtWriterException("Failed to close table writers");
            suppressed.forEach(e::addSuppressed);
            logger.warn("Closing writers failed, abort tx {}", writeTx, e);
            abortWriteTx();
            throw e;
        }

        if (cookies.isEmpty()) {
            // не пригодилась
            logger.debug("Nothing to commit, abort tx {}", writeTx);
            abortWriteTx();
            return;
        }

        logger.info("Writers closed, commit tx {}", writeTx);
        commitWriteTx();

        List<Long> cookiesForCommit = List.copyOf(cookies);
        cookies.clear();

        Exception exc = null;
        try {
            logger.info("Commit {} logbroker cookies", cookiesForCommit.size());
            logger.debug("Committing cookies: {}", cookiesForCommit);
            logbroker.commit(cookiesForCommit);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            exc = e;
        } catch (TimeoutException | RuntimeException e) {
            exc = e;
        }
        if (exc != null) {
            logger.error("Logbroker commit failed, but logs were written. Possible duplicates in {}", tables);
            throw new RuntimeException("Commit to logbroker failed", exc);
        }
    }

    @Override
    public void onShutdown() {
        interruptable.values().forEach(Runnable::run);
        shutdown = true;
    }

    @Override
    public void finish() {
        if (logbroker != null) {
            try {
                logger.info("Close logbroker client");
                logbroker.close();
                logger.info("Logbroker client closed");
            } catch (RuntimeException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
        if (writeTx != null) {
            logger.warn("Found uncommitted transaction {} — abort it", writeTx);
            try {
                abortWriteTx();
                logger.info("Transaction aborted");
            } catch (RuntimeException e) {
                logger.warn("Error while aborting transaction", e);
            }
        }
    }

    class TableOperator {
        private final YPath table;
        private final LocalDate logDate;
        private boolean exists;
        private boolean tableAttributesChecked;
        private TableWriter<BsExportLogRow> writer;
        private long bytesWrittenApprox;
        private MonotonicTime firstWriteAt;

        TableOperator(LocalDate logDate) {
            this.logDate = logDate;
            String date = logDate.format(DateTimeFormatter.ISO_LOCAL_DATE);
            table = YPath.simple(YtPathUtil.generatePath(LOGS_FOLDER, date));
        }

        String getTablePath() {
            return table.justPath().toString();
        }

        private void createTableIfNotExists() {
            if (!exists) {
                logger.debug("Check table {} for existing", table);
                exists = yt.runRpcCommandWithTimeout(yt.getYtClient()::existsNode, new ExistsNode(table));
                // если таблицу удалят "на ходу" — упасть при записи лучше и понятнее,
                // чем тихо создать таблицу обратно
                tableAttributesChecked = !exists;    // если таблица существовала — надо проверить ее атрибуты
            }
            if (exists) {
                logger.trace("Table {} exists or already checked", table);
                return;
            }

            LocalDateTime expirationTime = calcExpirationTime();
            long expiration = moscowDateTimeToInstant(expirationTime).toEpochMilli();
            CreateNode createNodeRequest = new CreateNode(table, ObjectType.Table)
                    .setIgnoreExisting(true)
                    .setRecursive(true)
                    .addAttribute(EXPIRATION_TIME_ATTR, YTree.longNode(expiration))
                    .addAttribute(SCHEMA_ATTR, serializer.getSchema().toYTree());

            logger.info("Create logs table {} with expirationTime {}", table.toString(), expirationTime);
            yt.runRpcCommandWithTimeout(yt.getYtClient()::createNode, createNodeRequest);
        }

        private void alterTableIfNeeded() {
            if (tableAttributesChecked) {
                logger.trace("Table {} schema and codec already checked", table);
                return;
            }

            GetNode getNodeReq = new GetNode(table)
                    .setAttributes(ColumnFilter.of(SORTED_ATTR, COMPRESSION_CODEC_ATTR, ERASURE_CODEC_ATTR));
            YTreeNode node = yt.runRpcCommandWithTimeout(yt.getYtClient()::getNode, getNodeReq);

            if (node.getAttributeOrThrow(SORTED_ATTR).boolValue()) {
                logger.info("Alter table {} to make it unsorted", table);
                AlterTable alterTableReq = new AlterTable(table.toString()).setSchema(serializer.getSchema());
                yt.runRpcCommandWithTimeout(yt.getYtClient()::alterTable, alterTableReq);
            }

            String desiredErasureCodec = "none";
            if (!node.getAttributeOrThrow(ERASURE_CODEC_ATTR).stringValue().equals(desiredErasureCodec)) {
                SetNode setNodeRequest = new SetNode(table.attribute(ERASURE_CODEC_ATTR),
                        YTree.stringNode(desiredErasureCodec));
                logger.info("Set {} attribute {} to {}", table, ERASURE_CODEC_ATTR, desiredErasureCodec);
                yt.runRpcCommandWithTimeout(yt.getYtClient()::setNode, setNodeRequest);
            }

            String desiredCompressionCodec = "lz4";
            if (!node.getAttributeOrThrow(COMPRESSION_CODEC_ATTR).stringValue().equals(desiredCompressionCodec)) {
                SetNode setNodeRequest = new SetNode(table.attribute(COMPRESSION_CODEC_ATTR),
                        YTree.stringNode(desiredCompressionCodec));
                logger.info("Set {} attribute {} to {}", table, COMPRESSION_CODEC_ATTR, desiredCompressionCodec);
                yt.runRpcCommandWithTimeout(yt.getYtClient()::setNode, setNodeRequest);
            }

            tableAttributesChecked = true;
        }

        private LocalDateTime calcExpirationTime() {
            LocalDateTime dayEnd = logDate.atStartOfDay().plusDays(1).minusSeconds(1);
            if (logsYtCluster == YtCluster.FREUD) {
                return dayEnd.plusWeeks(1);
            } else {
                return dayEnd.plusYears(3);
            }
        }

        int getAvailableChunksCount() {
            checkState(writeTx != null, "Write transaction is not available");
            return getAvailableChunksCount(writeTx::getNode);
        }

        int getAvailableChunksCount(Function<GetNode, CompletableFuture<YTreeNode>> getNodeFunc) {
            logger.trace("Calc chunks stat for {}", table);

            GetNode req = new GetNode(table).setAttributes(ColumnFilter.of("chunk_count", COMPRESSED_DATA_SIZE));

            YTreeNode node = yt.runRpcCommandWithTimeout(getNodeFunc, req);

            long chunkCount = node.getAttributeOrThrow("chunk_count").longValue();
            // тут неявно полагаемся, что работаем только с пожатыми таблицами (compression_codec != none)
            long size = node.getAttributeOrThrow(COMPRESSED_DATA_SIZE).longValue();

            // формулу надо подбирать практически, потому что без force_transform чанки недостаточно укрупняются
            long estimatedChunksCount = size / DESIRED_CHUNK_SIZE + 1;
            long borderChunksCount = estimatedChunksCount + CHUNKS_GAP;

            long avgChunkSize = size / (chunkCount + 1) / 1024 / 1024;
            int result = (int) (borderChunksCount - chunkCount);
            logger.info("Table {}.{} stat: size {}, chunks {}, average size {} (MB). Chunks available for write: {}",
                    logsYtCluster, table, size, chunkCount, avgChunkSize, result);

            return result;
        }

        boolean mergeTable() {
            // таймаут транзакции небольшой, потому что она пингуется при ожидании лока и операции
            StartTransaction startTransaction = StartTransaction.master()
                    .setTransactionTimeout(Duration.ofMinutes(1))
                    .setPingPeriod(null)
                    .setPing(false);

            var tx = yt.runRpcCommandWithTimeout(yt.getYtClient()::startTransaction, startTransaction);
            logger.info("Started merge transaction {}", tx);
            LockWrapper lockWrapper = yt.createLockWrapper(tx);
            TableMerger tableMerger = yt.createTableMerger(tx, table)
                    .withPool(MERGE_OPERATIONS_POOL)
                    .withOperationWeight(2)
                    .withDesiredChunkSize(DESIRED_CHUNK_SIZE);
            interruptable.put(lockWrapper, lockWrapper::interrupt);
            interruptable.put(tableMerger, tableMerger::interrupt);

            try {
                boolean acquired = lockWrapper.createLock(tableMerger.createLockNodeRequest())
                        .waitLock(Duration.ofSeconds(5), LOCK_WAIT_TIMEOUT);
                if (!acquired) {
                    logger.warn("Failed to acquire exclusive lock on table, abort tx {}", tx);
                    lockWrapper.abortTx();
                    return false;
                }

                if (getAvailableChunksCount(tx::getNode) > 0) {
                    // такое может быть, если мы встали в очередь за exclusive-локом не первыми
                    logger.info("Table was merged by another worker");
                    tableMerger.commitTx();
                    return true;
                }

                OperationWrapper.State last = tableMerger.doMerge(Duration.ofSeconds(10), MERGE_TIMEOUT);
                if (!last.isCompleted()) {
                    logger.warn("Merge was not succeeded, abort tx {}", tx);
                    tableMerger.abortTx();
                    return false;
                }

                if (getAvailableChunksCount(tx::getNode) > 10) {
                    logger.info("Successfully merged, commit tx {}", tx);
                    tableMerger.commitTx();
                    return true;
                }

                logger.warn("Size of the chunks in {} is not large enough. Doing a force_transform merge", table);
                tableMerger = tableMerger.newMerger().withForceTransform(true);
                last = tableMerger.doMerge(Duration.ofSeconds(10), MERGE_TIMEOUT);
                if (!last.isCompleted()) {
                    logger.warn("Second merge was not succeeded, abort tx {}", tx);
                    tableMerger.abortTx();
                    return false;
                }

                logger.info("Successfully merged in a second pass, commit tx {}", tx);
                tableMerger.commitTx();
                return true;
            } finally {
                interruptable.remove(tableMerger);
                interruptable.remove(lockWrapper);
            }
        }

        private void openWriterIfNeeded() {
            if (writer != null) {
                return;
            }

            WriteTable<BsExportLogRow> writeTableReq = new WriteTable<>(table.append(true).toString(), serializer);
            writer = yt.runRpcCommandWithTimeout(writeTx::writeTable, writeTableReq);
            bytesWrittenApprox = 0;
            firstWriteAt = clock.getTime();
        }

        void write(List<BsExportLogRow> rows, long bytesApprox) {
            openWriterIfNeeded();

            try {
                int attempt = 0;
                while (!writer.write(rows)) {
                    if (++attempt > 10) {
                        throw new YtWriterException("Write attempts exceeded");
                    }
                    logger.trace("Wait table writer {}", table);
                    try (TraceProfile ignored = Trace.current().profile("writer:waitReadyEvent")) {
                        getWithTimeout(writer.readyEvent(), WRITE_TIMEOUT, "Timeout waiting for data is written");
                    }
                }
            } catch (IOException e) {
                throw new YtWriterException("Failed to write data", e);
            }

            bytesWrittenApprox += bytesApprox;
        }

        boolean hasWriter() {
            return writer != null;
        }

        long getBytesWrittenApprox() {
            return bytesWrittenApprox;
        }

        MonotonicTime getFirstWriteTime() {
            return firstWriteAt;
        }

        void closeWriter() {
            logger.trace("Closing table writer: {}", table);
            if (writer == null) {
                // should not be here
                checkState(cookies.isEmpty(), "Writer is lost, but there is uncommitted cookies");
                checkState(writeTx == null, "Writer is lost, but there is uncommitted transaction");
                logger.trace("Writer already closed");
                return;
            }

            long megabytes = bytesWrittenApprox / 1024 / 1024;
            logger.info("Close writer for {}. Approximately {} megabytes written", table, megabytes);
            try (TraceProfile ignored = Trace.current().profile("writer:close")) {
                getWithTimeout(writer.close(), WRITE_TIMEOUT, "Error closing writer");
            } finally {
                writer = null;
                firstWriteAt = null;
            }
        }
    }
}
