package ru.yandex.direct.jobs.logs.deserializer;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.google.common.primitives.Ints;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jobs.logs.model.BsExportLogRow;

import static com.google.common.base.Preconditions.checkState;

public class BsExportLogDeserializer extends StdDeserializer<BsExportLogRow> {
    public static final TypeReference<List<BsExportLogRow>> TYPE_REFERENCE = new TypeReference<>() {
    };

    private static final Logger logger = LoggerFactory.getLogger(BsExportLogDeserializer.class);
    private final boolean optimizeMemory;

    public BsExportLogDeserializer(boolean optimizeMemory) {
        super(BsExportLogRow.class);
        this.optimizeMemory = optimizeMemory;
    }

    private static byte[] getSourceBytes(JsonParser p) {
        return (byte[]) p.getCurrentLocation().getSourceRef();
    }

    private static int[] getJsonValuePositions(JsonParser p) throws IOException {
        int[] result = new int[2];

        result[0] = getCurrentByteOffset(p);
        result[0] -= 1; // '{' — тоже включаем

        p.skipChildren();

        result[1] = getCurrentByteOffset(p);

        return result;
    }

    private static int getCurrentByteOffset(JsonParser p) {
        int pos = Ints.checkedCast(p.getCurrentLocation().getByteOffset());
        checkState(pos > 0, "source doesn't provide ByteOffset");
        return pos;
    }

    private static Long getValueAsLong(JsonParser p) throws IOException {
        JsonToken token = p.nextToken();
        switch (token) {
            case VALUE_NUMBER_INT:
                return p.getLongValue();
            case VALUE_STRING:
                return Long.parseLong(p.getText());
            case VALUE_NULL:
                return null;
            default:
                throw new JsonParseException(p,
                        "Unexpected token type for field " + p.getCurrentName() + " - " + token);
        }
    }

    private static String getValueAsString(JsonParser p) throws IOException {
        String name = p.getCurrentName();
        JsonToken token = p.nextToken();
        if (token == null) {
            return null;
        } else if (token.isScalarValue()) {
            return p.getText();
        } else {
            throw new JsonParseException(p, "Can't get value for field " + name);
        }
    }

    private static void nextTokenWithStartObjectCheck(JsonParser p) throws IOException {
        JsonToken token = p.nextToken();
        if (token != JsonToken.START_OBJECT) {
            throw new JsonParseException(p, "Expected begin of object");
        }
    }

    private BsExportLogRow deserialize(JsonParser p, BsExportLogRow row, int depth) throws IOException {
        checkState(depth++ < 3, "unexpected nested objects depth");

        long counter = 0;
        while (p.nextToken() != JsonToken.END_OBJECT) {
            checkState(counter++ < 1_000, "possible infinite loop detected");
            JsonToken currentToken = p.currentToken();
            String currentName = p.getCurrentName();

            // переход к значению (nextToken) везде делается внутри читающих методов

            if (currentToken == JsonToken.FIELD_NAME) {
                if ("cid".equals(currentName)) {
                    row.setCid(getValueAsLong(p));
                } else if ("pid".equals(currentName)) {
                    row.setPid(getValueAsLong(p));
                } else if ("level".equals(currentName)) {
                    row.setLevel(getValueAsString(p));
                } else if ("uuid".equals(currentName)) {
                    row.setUuid(getValueAsString(p));
                } else if ("iter_id".equals(currentName)) {
                    row.setIterId(getValueAsLong(p));
                } else if ("actual_in_direct_db_at".equals(currentName)) {
                    row.setActualInDirectDbAt(getValueAsLong(p));
                } else if ("data".equals(currentName)) {
                    nextTokenWithStartObjectCheck(p);
                    int[] pos = getJsonValuePositions(p);
                    byte[] source = getSourceBytes(p);
                    if (optimizeMemory) {
                        row.setDataPos(source, pos);
                    } else {
                        String data = new String(source, pos[0], pos[1] - pos[0], StandardCharsets.UTF_8);
                        row.setData(data);
                    }
                } else if ("debug_info".equals(currentName)) {
                    nextTokenWithStartObjectCheck(p);
                    deserialize(p, row, depth);
                } else if ("shard".equals(currentName)) { // здесь и ниже: debug_info
                    row.setShard(getValueAsLong(p));
                } else if ("reqid".equals(currentName)) {
                    row.setReqid(getValueAsLong(p));
                } else if ("par_norm_nick".equals(currentName)) {
                    row.setParType(getValueAsString(p));
                } else if ("host".equals(currentName)) {
                    row.setLogHost(getValueAsString(p));
                } else {
                    logger.trace("Skip field {}", currentName);
                }
            } else if (currentToken == JsonToken.START_ARRAY) {
                // на корневом уровне массивов у нас нет
                logger.trace("Unexpected array in root level field {}, skip it", currentName);
                p.skipChildren();
            } else if (currentToken == JsonToken.START_OBJECT) {
                // nested object
                logger.trace("Unexpected object in root level field {}, skip it", currentName);
                p.skipChildren();
            }
        }

        return row;
    }

    @Override
    public BsExportLogRow deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        var res = new BsExportLogRow();
        checkState(p.getCurrentLocation().getSourceRef() instanceof byte[], "expected bytearray as json source");
        return deserialize(p, res, 0);
    }
}
