package ru.yandex.direct.jobs.logs.serializer;

import java.util.List;

import ru.yandex.direct.jobs.logs.model.BsExportLogRow;
import ru.yandex.yt.ytclient.object.WireProtocolWriteable;
import ru.yandex.yt.ytclient.object.WireRowSerializer;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.yt.ytclient.tables.ColumnValueType.INT64;
import static ru.yandex.yt.ytclient.tables.ColumnValueType.STRING;

/**
 * Сериализация строчки лога bsexport.
 * <p>
 * Почему не MappedRowSerializer (для YTreeMapNode)?
 * Её из JSON'а тоже надо собирать руками (а если писать имена колонок и типы, то какая разница где).
 * Еще таким способом получается не выделить память под data для всех строчек сразу.
 * <p>
 * При удалении колонок: сначала объявить как required: false, зарелизить, дождаться создания новых таблиц,
 * только после этого можно удалять совсем
 * <p>
 * При добавлении колонок: сначала добавить в схему, но не добавлять
 * в {@link #serializeRow(BsExportLogRow, WireProtocolWriteable, boolean, int[])}.
 * Затем дождаться создания и записи в новые таблицы, и можно добавлять колонку в запись.
 * Не забыть поднять значение в {@code writeValueCount}
 */
public class BsExportLogWireSerializer implements WireRowSerializer<BsExportLogRow> {

    private static final List<YtColumnSpec<BsExportLogRow>> COLUMNS = List.of(
            new YtColumnSpec<>("cid", INT64, true, "meta", BsExportLogRow::getCid),
            new YtColumnSpec<>("pid", INT64, true, "meta", BsExportLogRow::getPid),
            new YtColumnSpec<>("log_time", STRING, true, "meta", BsExportLogRow::getLogTime),
            new YtColumnSpec<>("level", STRING, true, "meta", BsExportLogRow::getLevel),
            new YtColumnSpec<>("reqid", INT64, true, "meta", BsExportLogRow::getReqid),
            new YtColumnSpec<>("uuid", STRING, true, "meta", BsExportLogRow::getUuid),
            new YtColumnSpec<>("shard", INT64, true, "meta", BsExportLogRow::getShard),
            new YtColumnSpec<>("par_type", STRING, true, "meta", BsExportLogRow::getParType),
            new YtColumnSpec<>("iter_id", INT64, true, "meta", BsExportLogRow::getIterId),
            new YtColumnSpec<>("log_type", STRING, true, "meta", BsExportLogRow::getLogType),
            new YtColumnSpec<>("log_host", STRING, true, "meta", BsExportLogRow::getLogHost),
            new YtColumnSpec<>("data", STRING, true, "data", BsExportLogRow::getData),
            new YtColumnSpec<>("actual_in_direct_db_at", INT64, true, "meta", BsExportLogRow::getActualInDirectDbAt)
    );
    private static final TableSchema SCHEMA = buildSchema();

    private static TableSchema buildSchema() {
        TableSchema.Builder builder = TableSchema.builder();
        COLUMNS.stream().map(YtColumnSpec::getSchema).forEach(builder::add);
        return builder.setStrict(true)
                .setUniqueKeys(false)
                .build();
    }

    @Override
    public TableSchema getSchema() {
        return SCHEMA;
    }

    @Override
    public void serializeRow(BsExportLogRow row, WireProtocolWriteable writeable, boolean keyFieldsOnly,
                             boolean aggregate, int[] idMapping) {
        if (idMapping != null) {
            throw new UnsupportedOperationException("idMapping is not supported");
        }
        if (keyFieldsOnly) {
            throw new UnsupportedOperationException("keyFieldsOnly");
        }

        writeable.writeValueCount(COLUMNS.size());

        int idx = 0;
        for (var column : COLUMNS) {
            Object value = column.getValue(row);
            ColumnValueType type = column.getType();
            writeable.writeValueHeader(idx, type, false, type == STRING ? ((byte[]) value).length : 0);
            if (type == STRING) {
                writeable.onBytes((byte[]) value);
            } else if (type == INT64) {
                writeable.onInteger((Long) value);
            } else {
                throw new IllegalStateException("Unsupported type " + type);
            }
            idx++;
        }
    }

}
