package ru.yandex.direct.jobs.marketfeeds

import ru.yandex.direct.core.entity.feed.DEFAULT_REFRESH_INTERVAL
import ru.yandex.direct.core.entity.feed.createFakeFeedUrl
import ru.yandex.direct.core.entity.feed.model.BusinessType.RETAIL
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedRow
import ru.yandex.direct.core.entity.feed.model.FeedType.YANDEX_MARKET
import ru.yandex.direct.core.entity.feed.model.MasterSystem.MARKET
import ru.yandex.direct.core.entity.feed.model.Source.URL
import ru.yandex.direct.core.entity.feed.model.UpdateStatus.ERROR
import ru.yandex.direct.core.entity.feed.model.UpdateStatus.NEW
import ru.yandex.direct.model.ModelChanges

object FeedConverter {

    private const val FEED_NAME_BY_FEED_ID_PATTERN = "Фид из Яндекс.Маркета #%d"
    private const val FEED_NAME_BY_SHOP_NAME_PATTERN = "Фид из Яндекс.Маркета по магазину %s"
    const val DELETED_FEED_NAME = "DELETED"

    fun ytRowToFeed(row: FeedRow, isFeedByShopId: Boolean): Feed =
        Feed().apply {
            clientId = row.clientId
            marketBusinessId = row.businessId
            marketShopId = row.shopId
            // Для синхронизации фидов по магазину - фейковое id маркетного фида равно shop_id
            marketFeedId = if (isFeedByShopId) marketShopId else row.feedId
            businessType = RETAIL
            feedType = YANDEX_MARKET
            source = URL

            name = if (isFeedByShopId) {
                String.format(FEED_NAME_BY_SHOP_NAME_PATTERN, row.shopName)
            } else {
                String.format(FEED_NAME_BY_FEED_ID_PATTERN, row.feedId)
            }

            shopName = row.shopName
            login = row.login
            plainPassword = row.password
            val feedUrlMarketFeedId = if (!isFeedByShopId) row.feedId else null
            url = createFakeFeedUrl(row.businessId, row.shopId, feedUrlMarketFeedId, row.url, true, isFeedByShopId)
            masterSystem = MARKET
            updateStatus = NEW
            refreshInterval = DEFAULT_REFRESH_INTERVAL
        }

    fun feedsToModelChanges(feedIdByMarketFeedId: Map<Long, Long>, changedYtFeeds: List<Feed>) =
        changedYtFeeds.map {
            val feedId = feedIdByMarketFeedId[it.marketFeedId]!!
            ModelChanges(feedId, Feed::class.java).apply {
                process(it.marketBusinessId, Feed.MARKET_BUSINESS_ID)
                process(it.marketShopId, Feed.MARKET_SHOP_ID)
                process(it.name, Feed.NAME)
                process(it.url, Feed.URL)
                process(it.login, Feed.LOGIN)
                process(it.plainPassword, Feed.PLAIN_PASSWORD)
                process(it.updateStatus, Feed.UPDATE_STATUS)
                process(it.refreshInterval, Feed.REFRESH_INTERVAL)
                process(it.shopName, Feed.SHOP_NAME)
            }
        }

    fun feedIdToDeletedModelChanges(feedIds: List<Long>) =
        feedIds.map {
            ModelChanges(it, Feed::class.java).apply {
                process(DELETED_FEED_NAME, Feed.NAME)
                process(ERROR, Feed.UPDATE_STATUS)
                process(0L, Feed.REFRESH_INTERVAL)
            }
        }
}
