package ru.yandex.direct.jobs.marketfeeds

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.banner.service.BannerSuspendResumeService
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedRow
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.MasterSystem.MARKET
import ru.yandex.direct.core.entity.feed.repository.DataCampFeedYtRepository
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.repository.MarketFeedsConfig
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.model.UidAndClientId
import ru.yandex.direct.dbutil.sharding.ShardKey.CLIENT_ID
import ru.yandex.direct.dbutil.sharding.ShardSupport
import ru.yandex.direct.feature.FeatureName.MARKET_FEEDS_ALLOWED
import ru.yandex.direct.feature.FeatureName.SYNC_MARKET_FEEDS_BY_SHOP_ID
import ru.yandex.direct.jobs.marketfeeds.FeedConverter.DELETED_FEED_NAME
import ru.yandex.direct.jobs.marketfeeds.ytmodels.generated.YtFeeds
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.ytwrapper.client.YtExecutionUtil.executeWithFallback
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.direct.ytwrapper.model.YtTable
import java.util.EnumSet

private val LOGGER = LoggerFactory.getLogger(MarketFeedsUpdateService::class.java)

@Service
class MarketFeedsUpdateService @Autowired constructor(
    private val ytProvider: YtProvider,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val shardSupport: ShardSupport,
    private val featureService: FeatureService,
    private val feedRepository: FeedRepository,
    private val clientRepository: ClientRepository,
    private val adGroupRepository: AdGroupRepository,
    private val feedService: FeedService,
    private val bannerTypedRepository: BannerTypedRepository,
    private val bannerSuspendResumeService: BannerSuspendResumeService,
    private val dataCampFeedYtRepository: DataCampFeedYtRepository,
    marketFeedsConfig: MarketFeedsConfig
) {

    private val ytClusters: List<YtCluster> = marketFeedsConfig.clusters
    private val ytMarketFeedsTable: YtTable = YtTable(marketFeedsConfig.ytDataCampFeedsTablePath)

    class SyncResults {
        val addedIds = mutableListOf<Long>()
        val updatedIds = mutableListOf<Long>()
        val removedIds = ArrayList<Long>()
        val marketAsDeletedIds = mutableListOf<Long>()
        var errorsCount = 0
    }

    fun updateMarketFeeds(shard: Int): SyncResults? {
        // Пропертя включает синхронизацию маркетных фидов на всех, а если пропертёй синхронизация не включена,
        // то по фиче синхронизируются конкретные клиенты.
        val isSyncEnabled = ppcPropertiesSupport.get(PpcPropertyNames.MARKET_FEEDS_SYNC_ENABLED).getOrDefault(false)
        val featureExist = featureService.isFeatureExist(MARKET_FEEDS_ALLOWED)
        if (!isSyncEnabled && !featureExist) {
            return null
        }

        var clientIds = getAndFilterClientIds(shard, isSyncEnabled)
        val operatorUidToClient: Map<Long, UidAndClientId> = getOperatorUidToClient(shard, clientIds)
        clientIds = operatorUidToClient.values.map { it.clientId.asLong() }.toSet()
        val clientIdsWithFeedsByShop = getClientIdsWithFeedsByShop(clientIds)

        val ytFeedsByClientId: Map<Long, List<Feed>> = getYtFeeds(clientIds, clientIdsWithFeedsByShop)
        val dbFeedsByClientId = feedRepository.getAllDataCampFeedsSimple(shard).groupBy { it.clientId }

        val syncResults = SyncResults()
        for ((operatorUid, uidAndClientId) in operatorUidToClient) {
            val ytFeeds = ytFeedsByClientId.getOrDefault(uidAndClientId.clientId.asLong(), emptyList())
            val dbFeeds = dbFeedsByClientId.getOrDefault(uidAndClientId.clientId.asLong(), emptyList())
            syncClientFeeds(shard, operatorUid, uidAndClientId, ytFeeds, dbFeeds, syncResults)
        }

        return syncResults
    }

    private fun getOperatorUidToClient(shard: Int, clientIds: Set<Long>): Map<Long, UidAndClientId> {
        val ids: List<ClientId> = clientIds.mapTo(mutableListOf()) { ClientId.fromLong(it) }
        val clients = clientRepository.get(shard, ids)
        return clients.associate { client ->
            val operatorUid = client.agencyUserId ?: client.chiefUid
            operatorUid to UidAndClientId.of(client.chiefUid, ClientId.fromLong(client.id))
        }
    }

    private fun syncClientFeeds(
        shard: Int,
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        ytFeeds: List<Feed>,
        dbFeeds: List<FeedSimple>,
        syncResults: SyncResults
    ) {
        try {
            val dbMarketFeeds = dbFeeds.filter { it.masterSystem == MARKET }

            val newYtFeeds = filterNewFeeds(ytFeeds, dbFeeds)
            addFeeds(operatorUid, uidAndClientId, newYtFeeds, syncResults)

            val changedYtFeeds = filterChangedFeeds(ytFeeds, dbMarketFeeds)
            updateFeeds(operatorUid, uidAndClientId, dbFeeds, changedYtFeeds, syncResults)

            val removedFeeds = filterRemovedFeeds(ytFeeds, dbMarketFeeds)
            removeFeeds(shard, operatorUid, uidAndClientId, removedFeeds, syncResults)
        } catch (ex: RuntimeException) {
            when (ex) {
                // не ронянем всю синхронизацию на битых данных одного клиента
                is NullPointerException, is IllegalStateException, is IllegalArgumentException -> {
                    syncResults.errorsCount++
                    LOGGER.error("ClientId=${uidAndClientId.clientId}, Exception: $ex")
                }
                else -> throw ex
            }
        }
    }

    private fun removeFeeds(
        shard: Int,
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        removedFeeds: List<FeedSimple>,
        syncResults: SyncResults
    ) {
        val feedIds = removedFeeds.map { it.id }
        val adGroupIdsByFeedIdToStop = adGroupRepository.getAdGroupIdsByFeedId(
            shard, feedIds,
            EnumSet.of(AdGroupType.PERFORMANCE, AdGroupType.DYNAMIC)
        )
        val textAdGroupIdsByFeedId = adGroupRepository.getAdGroupIdsByFeedId(
            shard, feedIds,
            EnumSet.of(AdGroupType.BASE)
        )
        val usedFeedIds = adGroupIdsByFeedIdToStop.keys union textAdGroupIdsByFeedId.keys

        val clientUnusedFeedIds = feedIds.filter { !usedFeedIds.contains(it) }
        removeUnusedFeeds(operatorUid, uidAndClientId, clientUnusedFeedIds, syncResults)

        val clientUsedFeedIds = feedIds.filter { usedFeedIds.contains(it) }
        markAsDeletedUsedFeeds(operatorUid, uidAndClientId, clientUsedFeedIds, syncResults)

        val adGroupIds = clientUsedFeedIds
            .mapNotNull { adGroupIdsByFeedIdToStop[it] }
            .flatten()
        stopAds(shard, operatorUid, uidAndClientId, adGroupIds)
    }

    private fun stopAds(
        shard: Int, operatorUid: Long,
        uidAndClientId: UidAndClientId,
        adGroupIds: List<Long>
    ) {
        val banners = bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds)
        val bannerChanges = banners.map {
            ModelChanges(it.id, BannerWithSystemFields::class.java)
                .process(false, BannerWithSystemFields.STATUS_SHOW)
        }
        bannerSuspendResumeService.suspendResumeBanners(uidAndClientId.clientId, operatorUid, bannerChanges, false)
    }

    private fun markAsDeletedUsedFeeds(
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        clientUsedFeedIds: List<Long>,
        syncResults: SyncResults
    ) {
        val modelChanges = FeedConverter.feedIdToDeletedModelChanges(clientUsedFeedIds)
        val massResult = feedService.updateFeeds(uidAndClientId.clientId, uidAndClientId.uid, operatorUid, modelChanges)
        massResult.errors?.forEach { LOGGER.error("Error when mark a market feed as deleted, clientId:{${uidAndClientId.clientId}}, defect: {$it}") }
        val feedIds = massResult.result?.map { it.result } ?: emptyList()
        syncResults.marketAsDeletedIds.addAll(feedIds)
    }

    private fun removeUnusedFeeds(
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        clientUnusedFeedIds: List<Long>,
        syncResults: SyncResults
    ) {
        val massResult =
            feedService.deleteFeeds(uidAndClientId.clientId, uidAndClientId.uid, operatorUid, clientUnusedFeedIds)
        massResult.errors?.forEach { LOGGER.error("Error when delete a market feed, clientId:{${uidAndClientId.clientId}}, defect: {$it}") }
        val feedIds = massResult.result?.map { it.result } ?: emptyList()
        syncResults.removedIds.addAll(feedIds)
    }

    private fun updateFeeds(
        operatorUid: Long,
        uidAndClientId: UidAndClientId,
        dbFeeds: List<FeedSimple>,
        changedYtFeeds: List<Feed>,
        syncResults: SyncResults
    ) {
        val feedIdByMarketFeedId = dbFeeds.associateBy({ it.marketFeedId }, { it.id })
        val modelChanges = FeedConverter.feedsToModelChanges(feedIdByMarketFeedId, changedYtFeeds)
        val massResult = feedService.updateFeeds(uidAndClientId.clientId, uidAndClientId.uid, operatorUid, modelChanges)
        massResult.errors?.forEach { LOGGER.error("Error when update a market feed: clientId:{${uidAndClientId.clientId}}, defect: {$it}") }
        val feedIds = massResult.result?.map { it.result } ?: emptyList()
        syncResults.updatedIds.addAll(feedIds)
    }

    private fun addFeeds(
        operatorUid: Long, uidAndClientId: UidAndClientId, ytFeeds: List<Feed>,
        syncResults: SyncResults
    ) {
        val massResult = feedService.addFeeds(uidAndClientId.clientId, uidAndClientId.uid, operatorUid, ytFeeds)
        massResult.errors?.forEach { LOGGER.error("Error when add a new market feed: clientId:{${uidAndClientId.clientId}}, defect: {$it}") }
        val feedIds = massResult.result?.map { it.result } ?: emptyList()
        syncResults.addedIds.addAll(feedIds)
    }

    /**
     * Возвращает маркетные фиды, которые есть в базе, но по любой из причин для них нет соответствующего фида
     * в Маркете.
     * В выборку не попадают фиды, которые уже помечены в базе как удалённые (имеют имя DELETED).
     */
    private fun filterRemovedFeeds(ytFeeds: List<Feed>, currentFeeds: List<FeedSimple>): List<FeedSimple> {
        val dbMarketIds = currentFeeds.mapTo(HashSet()) { it.marketFeedId }
        val ytMarketIds = ytFeeds.mapTo(HashSet()) { it.marketFeedId }
        val removedMarketIds = dbMarketIds.minus(ytMarketIds)
        return currentFeeds.filter {
            removedMarketIds.contains(it.marketFeedId)
                // игнорим уже помеченные удалёнными фиды
                && DELETED_FEED_NAME != it.name
        }
    }

    /**
     * Выбирает фиды уже существующие в базе, но не совпадающие c ними по полям
     */
    private fun filterChangedFeeds(ytFeeds: List<Feed>, dbFeeds: List<FeedSimple>): List<Feed> {
        val ytFeedByMarketFeedId = ytFeeds.associateByTo(HashMap()) { it.marketFeedId }
        val changedYtFeeds = mutableListOf<Feed>()
        dbFeeds.forEach { dbFeed ->
            ytFeedByMarketFeedId[dbFeed.marketFeedId]
                ?.takeIf { ytFeed -> !hasMarketFeedIdCollision(ytFeed, dbFeed) }
                ?.takeIf { ytFeed -> ytFeed.hasChanges(dbFeed) }
                ?.let { ytFeed -> changedYtFeeds.add(ytFeed) }
        }
        return changedYtFeeds
    }

    //По идее метод всегда возвращает false, но данные приходят из активно развивающейся внешней системы...
    //И проверка дешёвая, а вот если прилетит "чёрный лебедь", то последствия довольно неприятные. Поэтому проверяем.
    private fun hasMarketFeedIdCollision(ytFeed: Feed, dbFeed: FeedSimple): Boolean {
        if (ytFeed.marketFeedId == dbFeed.marketFeedId && ytFeed.clientId != dbFeed.clientId) {
            LOGGER.error(
                "Feed has same marketFeedId=${ytFeed.marketFeedId} and different clientIds: " +
                    "${ytFeed.clientId},${dbFeed.clientId}."
            )
            return true
        }
        return false
    }

    private fun Feed.hasChanges(y: FeedSimple): Boolean =
        this.url != y.url
            || this.login != y.login
            || this.plainPassword != y.plainPassword
            || this.name != y.name
            || this.marketBusinessId != y.marketBusinessId
            || this.marketShopId != y.marketShopId
            || this.shopName != y.shopName

    private fun filterNewFeeds(ytFeeds: List<Feed>, dbFeeds: List<FeedSimple>): List<Feed> {
        val ytMarketIds = ytFeeds.mapTo(HashSet()) { it.marketFeedId }
        val dbMarketIds = dbFeeds.mapTo(HashSet()) { it.marketFeedId }
        val newMarketIds = ytMarketIds.minus(dbMarketIds)
        return ytFeeds.filter { newMarketIds.contains(it.marketFeedId) }
    }

    /**
     * Получить список идентификаторов клиентов, фиды которых нужно обработать на шарде [shard].
     * Достаем идентификаторы из YT и фильтруем:
     * - выбираем из шарда [shard]
     * - если синхронизация [isSyncEnabled] отключена, то выбираем таких, у которых включена фича [MARKET_FEEDS_ALLOWED]
     */
    private fun getAndFilterClientIds(shard: Int, isSyncEnabled: Boolean): Set<Long> {
        val clientIds = getAllClientIdsFromYt()
        val shardedClientIds = filterClientIdsByShard(shard, clientIds)
        return if (isSyncEnabled) {
            shardedClientIds
        } else {
            filterClientIdsByFeature(shardedClientIds)
        }
    }

    private fun getYtFeeds(clientIds: Set<Long>, clientIdsWithFeedsByShop: Set<Long>): Map<Long, List<Feed>> {
        val (shopFeedsByClient, feedsByClient) = dataCampFeedYtRepository.getYtFeedsRows(clientIds)
            .groupBy { it.clientId }
            .entries
            .partition { clientIdsWithFeedsByShop.contains(it.key) }

        val result = mutableMapOf<Long, List<Feed>>()

        shopFeedsByClient.forEach { entry ->
            val feeds = entry.value
                .groupBy { it.shopId }
                .entries
                .mapNotNull { it.value.minByOrNull(FeedRow::feedId) }
                .map { FeedConverter.ytRowToFeed(it, true) }
            result[entry.key] = feeds
        }

        feedsByClient.forEach { entry ->
            val feeds = entry.value
                .map { FeedConverter.ytRowToFeed(it, false) }
            result[entry.key] = feeds
        }

        return result
    }

    private fun filterClientIdsByFeature(clientIds: Collection<Long>): Set<Long> {
        val ids = clientIds.mapTo(HashSet()) { ClientId.fromLong(it) }
        // У фичи stateful-поведение, т.е. изменения от выкаченной на клиента фичи откатить назад можно только
        // миграцией. Плюс, для полноценной работы фиды клиентов должны удовлетворять некоторым условиям, поэтому,
        // раскатка данной фичи на процент пока не планируется.
        val isEnabledByClientId = featureService.isEnabledForClientIdsOnlyFromDb(ids, MARKET_FEEDS_ALLOWED.getName())
        return isEnabledByClientId.filterValues { it }
            .keys
            .mapTo(HashSet()) { it.asLong() }
    }

    private fun getClientIdsWithFeedsByShop(clientIds: Set<Long>): Set<Long> {
        val ids = clientIds.mapTo(HashSet()) { ClientId.fromLong(it) }
        return featureService.isEnabledForClientIds(ids, SYNC_MARKET_FEEDS_BY_SHOP_ID.getName())
            .filterValues { it }
            .keys
            .mapTo(HashSet()) { it.asLong() }
    }

    private fun filterClientIdsByShard(shard: Int, clientIds: List<Long>): Set<Long> {
        val shards = shardSupport.getShards(CLIENT_ID, clientIds)
        return clientIds.asSequence()
            .zip(shards.asSequence())
            .filter { it.second == shard }
            .mapTo(HashSet()) { it.first }
    }

    fun getAllClientIdsFromYt(): List<Long> =
        executeWithFallback(ytClusters, ytProvider::getOperator) {
            it.readTableField(
                ytMarketFeedsTable,
                YtFeeds.CLIENT_ID
            )
        }
            .filterNotNull()
            .distinct()

    fun getTableModificationTime(): String =
        executeWithFallback(ytClusters, ytProvider::getOperator) { it.readTableModificationTime(ytMarketFeedsTable) }
}
