package ru.yandex.direct.jobs.marketfeeds

import org.apache.commons.lang3.BooleanUtils.toBooleanObject
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.SYNC_MARKET_FEED_JOB_ENABLED
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.juggler.JugglerStatus.CRIT
import ru.yandex.direct.juggler.JugglerStatus.OK
import ru.yandex.direct.juggler.JugglerStatus.WARN
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.JugglerCheck.Duration
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1
import ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION
import ru.yandex.direct.juggler.check.model.CheckTag.YT
import ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_BUHTER
import ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_DMITANOSH
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob
import ru.yandex.direct.solomon.SolomonPushClient
import ru.yandex.direct.solomon.SolomonPushClientException
import ru.yandex.direct.solomon.SolomonUtils

const val LAST_TIME_PROPERTY_KEY_PATTERN = "market_feeds_last_processed_table_update_time_shard_%d"
private val LOGGER = LoggerFactory.getLogger(SyncMarketFeedsJob::class.java)

private const val SOLOMON_LABEL = "SyncMarketFeedsJob"

/**
 * Джоба синхронизирующая маркетные фиды с выгрузкой фидов из Маркета.
 */
@JugglerCheck(ttl = Duration(minutes = 60),
        needCheck = NonDevelopmentEnvironment::class,
        tags = [YT, JOBS_RELEASE_REGRESSION, DIRECT_PRIORITY_1],
        notifications = [OnChangeNotification(recipient = [LOGIN_BUHTER, LOGIN_DMITANOSH], method = [TELEGRAM], status = [OK, WARN, CRIT])])
@Hourglass(periodInSeconds = 600, needSchedule = NonDevelopmentEnvironment::class)
class SyncMarketFeedsJob @Autowired constructor(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val marketFeedsUpdateService: MarketFeedsUpdateService,
    private val solomonPushClient: SolomonPushClient
) : DirectShardedJob() {

    override fun execute() {
        val jobEnabledPropValue = ppcPropertiesSupport[SYNC_MARKET_FEED_JOB_ENABLED.getName()]
        val jobEnabled = jobEnabledPropValue?.let { toBooleanObject(it) } ?: false
        if (!jobEnabled) {
            LOGGER.info("Skip processing. Job is not enabled. $SYNC_MARKET_FEED_JOB_ENABLED=$jobEnabledPropValue")
            return
        }

        val tableModificationTime = marketFeedsUpdateService.getTableModificationTime()
        val lastSuccessTimeKey = String.format(LAST_TIME_PROPERTY_KEY_PATTERN, shard)
        val propertyValue = ppcPropertiesSupport[lastSuccessTimeKey]
        if (tableModificationTime == propertyValue) {
            LOGGER.info("Skip processing. YT-table hasn't been changed since the last time. $lastSuccessTimeKey=$propertyValue")
            return
        }

        val syncResults = marketFeedsUpdateService.updateMarketFeeds(shard)

        if (syncResults != null) {
            sendMetricsToSolomon(syncResults)
        }

        LOGGER.info("All market feeds on shard=$shard were updated.")
        ppcPropertiesSupport[lastSuccessTimeKey] = tableModificationTime
    }

    private fun sendMetricsToSolomon(
        syncResult: MarketFeedsUpdateService.SyncResults
    ) {
        LOGGER.info("Sending sync results to Solomon")

        val registry = SolomonUtils.newPushRegistry("flow", SOLOMON_LABEL, "shard", shard.toString())
        registry.gaugeInt64("added_feeds_count").set(syncResult.addedIds.size.toLong())
        registry.gaugeInt64("updated_feeds_count").set(syncResult.updatedIds.size.toLong())
        registry.gaugeInt64("removed_feeds_count").set(syncResult.removedIds.size.toLong())
        registry.gaugeInt64("marked_as_deleted_feeds_count").set(syncResult.marketAsDeletedIds.size.toLong())
        registry.gaugeInt64("errors_count").set(syncResult.errorsCount.toLong())

        try {
            solomonPushClient.sendMetrics(registry)
        } catch (e: SolomonPushClientException) {
            LOGGER.error("Got exception on sending metrics", e)
        }
    }

}
