package ru.yandex.direct.jobs.misprints

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.env.NonDevelopmentEnvironment
import ru.yandex.direct.jobs.misprints.service.ImportMisprintFixlistService
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.FunctionalUtils
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtCluster
import ru.yandex.inside.yt.kosher.cypress.YPath
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneOffset
import javax.annotation.ParametersAreNonnullByDefault

private val DEFAULT_LAST_IMPORTED_DATE = LocalDate.of(2021, 1, 1)
private val logger = LoggerFactory.getLogger(ImportMisprintFixlistJob::class.java)

/**
 * Джоба для импорта таблицы с популярными опечатками, за которые Модерация может отклонять.
 * Сама по себе таблица генерится Модерацией. Размер у нее небольшой — до 1000 записей
 * Будем раз в день проверять обновилась ли таблица, и если да, то привозить себе изменения
 * За обновлением будем следить с помощью
 * [свойства][ru.yandex.direct.common.db.PpcPropertyNames.MISPRINT_FIXLIST_LAST_IMPORT_DATE].
 */
@JugglerCheck(ttl = JugglerCheck.Duration(days = 5), needCheck = NonDevelopmentEnvironment::class)
@Hourglass(periodInSeconds = 60 * 60 * 24, needSchedule = NonDevelopmentEnvironment::class)
@ParametersAreNonnullByDefault
class ImportMisprintFixlistJob @Autowired constructor(private val ppcPropertiesSupport: PpcPropertiesSupport,
                                                      private val misprintFixlistService: ImportMisprintFixlistService,
                                                      directConfig: DirectConfig, private val ytProvider: YtProvider) : DirectJob() {
    private val clusters: List<YtCluster>
    private val yPath: YPath
    override fun execute() {
        logger.info("START")
        val lastImportedDate = ppcPropertiesSupport.get(PpcPropertyNames.MISPRINT_FIXLIST_LAST_IMPORT_DATE)
                .getOrDefault(DEFAULT_LAST_IMPORTED_DATE)
        val ytClusterToUpdate = getClusterWithUpdatedTable(lastImportedDate)
        if (ytClusterToUpdate != null) {
            logger.info("Found new table to import in " + ytClusterToUpdate.getName())
            misprintFixlistService.updateMisprintFixlist(ytClusterToUpdate, yPath)
            ppcPropertiesSupport.get(PpcPropertyNames.MISPRINT_FIXLIST_LAST_IMPORT_DATE).set(LocalDate.now())
        }
        logger.info("END")
    }

    /**
     * Возвращает cluster c самыми свежими данными или null, если таблица не обновлялась с последнего запуска джобы.
     */
    private fun getClusterWithUpdatedTable(lastImportedDate: LocalDate): YtCluster? {
        var lastUpdated = lastImportedDate.atStartOfDay().toEpochSecond(ZoneOffset.UTC)
        var ytClusterToUpdate: YtCluster? = null
        for (cluster in clusters) {
            val tableModificationTimeString = ytProvider[cluster].cypress()[yPath.attribute("modification_time")]
                    .stringValue()
            val tableModificationTime = Instant.parse(tableModificationTimeString).epochSecond
            if (tableModificationTime > lastUpdated) {
                lastUpdated = tableModificationTime
                ytClusterToUpdate = cluster
            }
        }
        return ytClusterToUpdate
    }

    init {
        val config = directConfig.getBranch("misprints")
        val clusterNames = config.getStringList("clusters")
        clusters = FunctionalUtils.mapList(clusterNames) { name: String? -> YtCluster.parse(name) }
        yPath = YPath.simple(config.getString("table_path"))
    }
}
