package ru.yandex.direct.jobs.moatexport;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.banner.type.measurers.BannerMeasurersRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.jobs.moatexport.MoatUtils.uploadToFtp;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

/**
 * Формирует выгрузку по баннерам с moat в csv файл. Выгрузка строится по баннерам, измерителем которых указан moat,
 * и у кампаний которых были показы за последние 8 месяцев, содержит в записи 8 колонок: id клиента, логин шефа клиента,
 * id кампании, имя кампании, id группы, имя группы, id креатива(баннера), имя креатива(баннера).
 * Данные выгружаются файлом на ftp Moat'а под именем "moat_reports_YYYY-MM-DD".
 * !!! Для локального запуска джобы нужно будет поднять локальный ftp сервер и ssh демон. Инструкцию для Ubuntu
 * оставил в resources/moatexport/setup_ftp_sftp.txt (список команд, как скрипт не отработает). В результате запуска
 * джобы должен появиться .csv файл с выгрузкой по пути '/home/sftpuser/'.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(days = 1),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, CheckTag.DIRECT_PRODUCT_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_AMMSAID,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 0 10,22 * * ?", needSchedule = ProductionOnly.class)
@ParametersAreNonnullByDefault
public class MoatBannersUploadJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(MoatBannersUploadJob.class);

    private static final int MAX_EXPECTED_SIZE_FOR_SHARD_IN_BYTES = 5 * 1024 * 1024;
    private static final int MAX_TIME_FROM_LAST_SHOW_IN_MONTHS = 8;

    private final ShardHelper shardHelper;
    private final BannerMeasurersRepository bannerMeasurersRepository;
    private final DirectConfig moatExportConfig;

    @Autowired
    public MoatBannersUploadJob(ShardHelper shardHelper,
                                BannerMeasurersRepository bannerMeasurersRepository,
                                DirectConfig directConfig) {
        this.shardHelper = shardHelper;
        this.bannerMeasurersRepository = bannerMeasurersRepository;
        this.moatExportConfig = directConfig.getBranch("moat_export");
    }

    @Override
    public void execute() {
        LocalDateTime currentDateTime = LocalDateTime.now();
        LocalDateTime dateTimeFrom = currentDateTime.minusMonths(MAX_TIME_FROM_LAST_SHOW_IN_MONTHS);
        try {
            String csvForUpload = collectMoatBannersToCsv(dateTimeFrom);
            uploadCsvToFtp(csvForUpload, currentDateTime);
            logger.info("Finish moat banners export to ftp");
        } catch (Exception ex) {
            logger.error("Error while getting banners or ftp params", ex);
            setJugglerStatus(JugglerStatus.CRIT, "Error while getting banners or ftp params");
        }
    }

    private String collectMoatBannersToCsv(LocalDateTime dateTimeFrom) {
        TraceProfile collectBannersProfile = Trace.current().profile("MoatBannersUploadJob:collectMoatBannersToCsv");
        String resultCsv = "";
        List<Integer> shardsForAlert = new ArrayList<>();
        int firstShard = shardHelper.dbShards().get(0);
        for (int shard : shardHelper.dbShards()) {
            try (TraceProfile ignored = Trace.current().profile(String.format("MoatBannersUploadJob" +
                    ":getMoatBannersFromShard_%d", shard));) {
                String bannersCsvForShard = bannerMeasurersRepository
                        .getBannersWithMoatCsv(shard, dateTimeFrom, shard == firstShard);
                resultCsv = resultCsv.concat(bannersCsvForShard);

                if (bannersCsvForShard.getBytes().length > MAX_EXPECTED_SIZE_FOR_SHARD_IN_BYTES) {
                    logger.warn("Data from shard {} has {} bytes.", shard, bannersCsvForShard.getBytes().length);
                    shardsForAlert.add(shard);
                }
            }
        }
        collectBannersProfile.close();

        if (!shardsForAlert.isEmpty()) {
            setJugglerStatus(JugglerStatus.WARN, "Unexpected large data size (>5Mb) in shards: " + shardsForAlert);
        }
        return resultCsv;
    }

    /**
     * Загрузить данные о баннерах с moat на партнерский ftp.
     *
     * @param csvForUpload    данные о баннерах в csv формате
     * @param currentDateTime текущая дата (для суффикса имени загружаемого файла)
     */
    private void uploadCsvToFtp(String csvForUpload, LocalDateTime currentDateTime) {
        String host = moatExportConfig.getString("host");
        int port = moatExportConfig.getInt("port");
        String path = moatExportConfig.getString("directory_path");
        String username = moatExportConfig.getString("username");
        String password = LiveResourceFactory.get(moatExportConfig.getString("password_path")).getContent();
        String filename = String.format("moat_reports_%s.csv",
                DateTimeFormatter.ISO_LOCAL_DATE.format(currentDateTime));

        try {
            uploadToFtp(host, port, username, password, path, filename, csvForUpload);
        } catch (Exception ex) {
            logger.error("Can't upload data", ex);
            setJugglerStatus(JugglerStatus.CRIT, "Can't upload data");
        }
    }

}
