package ru.yandex.direct.jobs.moatexport;

import java.io.ByteArrayOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.annotation.ParametersAreNonnullByDefault;

import com.univocity.parsers.csv.CsvWriter;
import com.univocity.parsers.csv.CsvWriterSettings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YqlQuery;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yql.ResultSetFuture;
import ru.yandex.yql.YqlConnection;

import static ru.yandex.direct.common.db.PpcPropertyNames.MOAT_PAGES_UPLOAD_STATUS;
import static ru.yandex.direct.jobs.moatexport.MoatUtils.uploadToFtp;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;
import static ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion.SQLv1;

@ParametersAreNonnullByDefault
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 150),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.LOGIN_OXID,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.WARN, JugglerStatus.CRIT}
        )
)
@Hourglass(cronExpression = "0 15 * * * ?", needSchedule = TypicalEnvironment.class)
public class MoatPagesJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(MoatPagesJob.class);

    private final YtProvider ytProvider;
    private static final String REQUEST_PATH = "classpath:///moatpages/moatpages.yql";
    private final String query;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private final CsvWriterSettings csvWriterSettings = new CsvWriterSettings();
    private final DirectConfig moatExportConfig;


    public MoatPagesJob(YtProvider ytProvider, PpcPropertiesSupport ppcPropertiesSupport, DirectConfig directConfig) {
        this.ytProvider = ytProvider;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.query = LiveResourceFactory.get(REQUEST_PATH).getContent();
        this.moatExportConfig = directConfig.getBranch("moat_export");
    }

    Future<?> checkOperation(String operationId) throws SQLException {
        YqlConnection connection = (YqlConnection) ytProvider.getYql(YtCluster.HAHN, SQLv1).getConnection();
        return connection.restoreResultSetFuture(operationId, "");
    }


    String runUpload(LocalDateTime date) {
        var ytOperator = this.ytProvider.getOperator(YtCluster.HAHN);

        var formattedDate = date.truncatedTo(ChronoUnit.DAYS).format(formatter);

        logger.info("Running YQL. Chevent log day {}", formattedDate);
        var future = ytOperator.yqlQueryBegin(new YqlQuery(this.query, formattedDate));

        ppcPropertiesSupport.get(MOAT_PAGES_UPLOAD_STATUS).set(Map.of(
                "operationId", future.getOperationId(),
                "date", formattedDate
        ));

        return future.getOperationId();
    }

    boolean tryToSaveResult(String operationId, String date) throws SQLException, ExecutionException {
        Future<?> rc = null;

        try {
            rc = checkOperation(operationId);
        } catch (SQLException throwables) {
            logger.warn("CheckOperation failed");
            return false;
        }

        if (rc.isDone()) {
            try {
                var result = saveResults(rc);

                uploadCsvToFtp(result, LocalDateTime.now());

                ppcPropertiesSupport.get(MOAT_PAGES_UPLOAD_STATUS).set(Map.of(
                        "operationId", operationId,
                        "date", date,
                        "isFinished", "1"
                ));

            } catch (InterruptedException e) {
                logger.error("Work was interrupted: {} ", operationId);
                return false;
            } catch (SQLException | ExecutionException e) {
                logger.error("Saving results failed: {} ", operationId);
                throw e;
            }
        } else {
            logger.info("Operation still in progress, operationId: {} ", operationId);
            return false;
        }

        return true;
    }

    /**
     * Загрузить данные о баннерах с moat на партнерский ftp.
     *
     * @param csvForUpload    данные о баннерах в csv формате
     * @param currentDateTime текущая дата (для суффикса имени загружаемого файла)
     */
    private void uploadCsvToFtp(String csvForUpload, LocalDateTime currentDateTime) {
        String host = moatExportConfig.getString("host");
        int port = moatExportConfig.getInt("port");
        String path = moatExportConfig.getString("directory_path");
        String username = moatExportConfig.getString("username");
        String password = LiveResourceFactory.get(moatExportConfig.getString("password_path")).getContent();
        String filename = String.format("placements__%s.csv", DateTimeFormatter.ISO_LOCAL_DATE.format(currentDateTime));

        try {
            uploadToFtp(host, port, username, password, path, filename, csvForUpload);
        } catch (Exception ex) {
            logger.error("Can't upload data", ex);
            setJugglerStatus(JugglerStatus.CRIT, "Can't upload data");
        }
    }

    @Override
    public void execute() {
        var lastStatus = ppcPropertiesSupport.get(MOAT_PAGES_UPLOAD_STATUS).get();

        if (lastStatus == null) {
            runUpload(LocalDateTime.now().minusDays(1));
        } else {
            try {
                if (lastStatus.containsKey("isFinished") || tryToSaveResult(lastStatus.get("operationId"),
                        lastStatus.get("date"))) {
                    var nextDayToRun = getNextDayToRun(lastStatus.get("date"));
                    if (nextDayToRun != null) {
                        runUpload(nextDayToRun);
                    }
                }
            } catch (SQLException | ExecutionException throwables) {
                logger.info("Restarting upload");
                var restartDay = LocalDateTime.of(LocalDate.parse(lastStatus.get("date"), formatter),
                        LocalTime.MIDNIGHT);
                runUpload(restartDay);
            }

        }

    }

    private String saveResults(Future<?> rc) throws SQLException, ExecutionException, InterruptedException {
        var resultSetFuture = (ResultSetFuture) rc;

        ByteArrayOutputStream csvResult = new ByteArrayOutputStream();
        Writer outputWriter = new OutputStreamWriter(csvResult);
        CsvWriter writer = new CsvWriter(outputWriter, csvWriterSettings);

        String[] header = {"PageId", "Domain"};
        writer.writeHeaders(header);

        try {
            ResultSet rs = resultSetFuture.get();

            while (rs.next()) {
                writer.writeRow(List.of(rs.getLong("pageid"), rs.getString("domain")));
            }

            writer.close();
            return csvResult.toString();
        } catch (InterruptedException | SQLException | ExecutionException e) {
            logger.error("Some exception occured", e);
            throw e;
        }
    }

    private LocalDateTime getNextDayToRun(String date) {
        var possibleNextDay = LocalDate.parse(date, formatter).atStartOfDay().plusDays(1);
        var today = LocalDateTime.now().truncatedTo(ChronoUnit.DAYS);

        if (possibleNextDay.isAfter(today)) {
            return null;
        }
        return today;
    }
}
