package ru.yandex.direct.jobs.mobileappsiosskadnetworkslots;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.mobileapp.model.SkAdNetworkSlot;
import ru.yandex.direct.core.entity.mobileapp.service.GrutSkadNetworkSlotService;
import ru.yandex.direct.core.entity.mobileapp.service.IosSkAdNetworkSlotManager;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static ru.yandex.direct.common.db.PpcPropertyNames.UPDATE_SKAD_NETWORK_SLOTS_IN_GRUT;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRODUCT_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба обновляет слоты для SkadNetwork
 * 1) освобождение занятых слотов: приложение потеряло верификацию, кампания, занимавшая слот, не показывается
 * 2) занятие свободных слотов: после прохождения верификации приложением, или после освобождения слота остановленной
 * кампанией
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        tags = {DIRECT_PRIORITY_0, DIRECT_PRODUCT_TEAM, JOBS_RELEASE_REGRESSION})
@Hourglass(cronExpression = "0 0/30 * * * ?")
@ParametersAreNonnullByDefault
public class MobileAppsIosSkadNetworkSlotsJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(MobileAppsIosSkadNetworkSlotsJob.class);
    private static final Duration CAMPAIGN_LIFITIME = Duration.ofDays(3);
    private static final int BATCH_SIZE = 500;

    private final RmpCampaignRepository rmpCampaignRepository;
    private final IosSkAdNetworkSlotManager iosSkAdNetworkSlotManager;
    private final CampaignRepository campaignRepository;
    private final PpcProperty<Boolean> updateSkadNetworkSlotsInGrut;

    public MobileAppsIosSkadNetworkSlotsJob(
            RmpCampaignRepository rmpCampaignRepository,
            IosSkAdNetworkSlotManager iosSkAdNetworkSlotManager,
            CampaignRepository campaignRepository,
            GrutSkadNetworkSlotService grutSkadNetworkSlotService,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.rmpCampaignRepository = rmpCampaignRepository;
        this.iosSkAdNetworkSlotManager = iosSkAdNetworkSlotManager;
        this.campaignRepository = campaignRepository;
        this.updateSkadNetworkSlotsInGrut = ppcPropertiesSupport.get(UPDATE_SKAD_NETWORK_SLOTS_IN_GRUT);
    }

    @Override
    public void execute() {
        var updateInGrut = updateSkadNetworkSlotsInGrut.getOrDefault(false);
        freeDeletedCampaignSlots(updateInGrut);
        freeUnAllowedCampaignSlots(updateInGrut);
        allocateFreeSlots(updateInGrut);
    }

    private void allocateFreeSlots(boolean updateInGrut) {
        var lastId = 0L;

        while (true) {
            var campaignIdsToBundleIds = rmpCampaignRepository.getCampaignsWithVerificationFlagIOS(
                    getShard(), true, lastId, BATCH_SIZE);
            logger.info("Collected {} campaigns without slots", campaignIdsToBundleIds.entrySet().size());
            if (campaignIdsToBundleIds.isEmpty()) {
                return;
            }

            var campaignIds = flatMapToSet(campaignIdsToBundleIds.values(), cId -> cId);
            var slots = iosSkAdNetworkSlotManager.getAllocatedSlotsByCampaignIds(campaignIds);
            var campaignIdsWithSlots = listToSet(slots, SkAdNetworkSlot::getCampaignId);

            for (var bundleId : campaignIdsToBundleIds.keySet()) {
                for (var campaignId : campaignIdsToBundleIds.get(bundleId)) {
                    if (campaignIdsWithSlots.contains(campaignId)) {
                        logger.info("Campaign {} already allocates slot", campaignId);
                        continue;
                    }

                    var slot = iosSkAdNetworkSlotManager.allocateCampaignSlot(bundleId, campaignId, updateInGrut);
                    if (slot != null) {
                        logger.info("Campaign {} allocated slot {} in bundle {}", campaignId, slot, bundleId);
                        campaignRepository.resetBannerSystemSyncStatus(getShard(), List.of(campaignId));
                        logger.info("BsSynced reset for campaign {}", campaignId);
                    } else {
                        logger.info("Campaign {} couldn't allocate slot in bundle {}", campaignId, bundleId);
                        // Выходим из цикла, так как если для бандла вернулся слот null, то все слоты заняты,
                        // и нет смысла дальше пытаться
                        break;
                    }
                }
            }
            lastId = campaignIds.stream().max(Long::compare).get();
        }
    }

    private void freeDeletedCampaignSlots(boolean updateInGrut) {
        for (var offset = 0L; ; offset += BATCH_SIZE) {
            var deletedCampaigns = iosSkAdNetworkSlotManager.getDeletedCampaignsWithAllocatedSlots(offset, BATCH_SIZE);
            if (deletedCampaigns.isEmpty()) {
                break;
            }
            logger.info("{} slot allocated for deleted campaigns", deletedCampaigns.size());
            freeSlotsByCampaignIds(deletedCampaigns, updateInGrut);
        }
    }

    private void freeUnAllowedCampaignSlots(boolean updateInGrut) {
        var timeThreshold = LocalDateTime.now().minus(CAMPAIGN_LIFITIME);
        for (var offset = 0L; ; offset += BATCH_SIZE) {
            var slotAllocatedCampaignIds =
                    iosSkAdNetworkSlotManager.getCampaignsWithAllocatedSlots(getShard(), offset, BATCH_SIZE);
            logger.info("Collected {} slot allocated campaigns", slotAllocatedCampaignIds.size());
            if (slotAllocatedCampaignIds.isEmpty()) {
                break;
            }

            var unAllowedSlotCampaigns = rmpCampaignRepository.getSkadUnAllowedCampaigns(getShard(), timeThreshold,
                    slotAllocatedCampaignIds);
            if (!unAllowedSlotCampaigns.isEmpty()) {
                logger.info("{} slot allocated campaigns have not been SkadNetwork allowed",
                        unAllowedSlotCampaigns.size());
                freeSlotsByCampaignIds(unAllowedSlotCampaigns, updateInGrut);
            }
        }
    }

    private void freeSlotsByCampaignIds(List<Long> stoppedCampaignIds, boolean updateInGrut) {
        var slots = iosSkAdNetworkSlotManager.getAllocatedSlotsByCampaignIds(stoppedCampaignIds);
        if (slots.isEmpty()) {
            return;
        }

        var campaignIds = mapList(slots, SkAdNetworkSlot::getCampaignId);
        logger.info("Collected {} campaigns to free slots", campaignIds);

        iosSkAdNetworkSlotManager.freeSlotsByCampaignIds(campaignIds, updateInGrut);
        campaignRepository.resetBannerSystemSyncStatus(getShard(), campaignIds);
    }
}
