package ru.yandex.direct.jobs.mobileappssync;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BinaryOperator;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Stopwatch;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTable;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTablesConfig;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TypicalEnvironment;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.transfermanagerutils.TransferManager;
import ru.yandex.direct.transfermanagerutils.TransferManagerConfig;
import ru.yandex.direct.transfermanagerutils.TransferManagerException;
import ru.yandex.direct.transfermanagerutils.TransferManagerJobConfig;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.direct.ytwrapper.YtPathUtil;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtSQLSyntaxVersion;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.cypress.Cypress;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtException;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeBooleanNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeMapNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.specs.CommandSpec;
import ru.yandex.inside.yt.kosher.operations.specs.JobIo;
import ru.yandex.inside.yt.kosher.operations.specs.ReduceSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.tables.types.JacksonTableEntryType;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static java.util.Map.entry;
import static java.util.function.Function.identity;
import static ru.yandex.direct.common.db.PpcPropertyNames.MOBILE_APPS_SYNC_JOB_ENABLED;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_API_TEAM;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.juggler.check.model.CheckTag.JOBS_RELEASE_REGRESSION;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.YtUtils.LAST_UPDATE_TIME_ATTR;

/**
 * Обновляет динамическую таблицу с данными о мобильных приложениях в Yt.
 * <p>
 * Методика:<ul>
 * <li>распарсить данные из исходных таблиц, имеющих вид "ключ" - "json"</li>
 * <li>отсортировать данные по ключу и подготовить их к переносу в динтаблицу</li>
 * <li>перенести подготовленные данные на кластер, где есть квота на динтаблицы, через ТМ</li>
 * <li>добавить таблице атрибут dynamic</li>
 * <li>подменить исходную таблицу свежей</li>
 * </ul>
 * Использует YQL, запрос лежит в ресурсах.
 * <p>
 * Примерное время выполнения - 30-45 минут на 1 магазин, сильно зависит от YQL и ТМ
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = ProductionOnly.class,
        //PRIORITY: выгрузка данных о мобильных приложениях из авроры в таблицы Директа, она не особо реалтаймовая,
        // но если она будет ломаться возникнут жалобы с проблемами добавления моб. приложений
        tags = {DIRECT_PRIORITY_1, DIRECT_API_TEAM},
        notifications = @OnChangeNotification(
                recipient = NotificationRecipient.CHAT_API_MONITORING,
                method = NotificationMethod.TELEGRAM,
                status = {JugglerStatus.OK, JugglerStatus.CRIT}
        )
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 200),
        needCheck = NonProductionEnvironment.class,
        //PRIORITY: выгрузка данных о мобильных приложениях из авроры в таблицы Директа, она не особо реалтаймовая,
        // но если она будет ломаться возникнут жалобы с проблемами добавления моб. приложений
        tags = {DIRECT_PRIORITY_1, DIRECT_API_TEAM, JOBS_RELEASE_REGRESSION}
)
@Hourglass(periodInSeconds = 60 * 60, needSchedule = TypicalEnvironment.class)
@ParametersAreNonnullByDefault
public class MobileAppsSyncJob extends DirectJob {
    private static final Logger logger = LoggerFactory.getLogger(MobileAppsSyncJob.class);
    private static final Duration WAIT_MOUNT_DURATION = Duration.ofMinutes(1);
    private static final Duration MAX_SORT_DURATION = Duration.ofHours(1);
    private static final Duration MAX_REPLACE_TOTAL_DURATION = Duration.ofMinutes(5);
    private static final Duration TM_MAX_AWAIT_DURATION = Duration.ofMinutes(30);
    private static final String ROW_COUNT_ATTRIBUTE = "row_count";
    static final String CREATION_TIME_ATTRIBUTE = "creation_time";
    static final String MODIFICATION_TIME_ATTRIBUTE = "modification_time";
    static final String LINK_TARGET_PATH = "&/@target_path";

    private final YtProvider ytProvider;
    private final MobileAppsSyncJobConfig config;
    private final TransferManager transferManager;
    private final MobileContentYtTablesConfig outputConfig;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public MobileAppsSyncJob(YtProvider ytProvider, DirectConfig directConfig,
                             TransferManagerConfig transferManagerConfig,
                             MobileContentYtTablesConfig outputConfig,
                             PpcPropertiesSupport ppcPropertiesSupport) {
        this.ytProvider = ytProvider;
        this.config = new MobileAppsSyncJobConfig(directConfig);
        this.transferManager = new TransferManager(transferManagerConfig);
        this.outputConfig = outputConfig;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    MobileAppsSyncJobConfig getConfig() {
        return config;
    }

    @Override
    public void execute() {
        if (!isJobEnabled()) {
            logger.info("Job is disabled, exiting");
            return;
        }

        TableSchema tableSchema;
        try {
            tableSchema = TableSchema.fromYTree(YTreeTextSerializer.deserialize(
                    new ClassPathResourceInputStreamSource("mobileappssync/schema.yson").getInput()));
        } catch (IOException e) {
            logger.error("Unable to read table schema", e);
            throw new TableSchemaParsingException(e);
        }

        boolean failed = false;
        for (MobileAppsSyncJobConfig.Task task : config.getTasks()) {
            try {
                taskWorkflow(tableSchema, task);
            } catch (MobileAppsSyncException e) {
                logger.error("Task failed: " + task.toString(), e);
                failed = true;
            }
        }

        if (failed) {
            throw new MobileAppsSyncException("At least some tasks failed");
        }
    }

    private boolean isJobEnabled() {
        return ppcPropertiesSupport.get(MOBILE_APPS_SYNC_JOB_ENABLED).getOrDefault(false);
    }

    /**
     * Обработка одной задачи из конфига
     */
    private void taskWorkflow(TableSchema tableSchema, MobileAppsSyncJobConfig.Task task)
            throws MobileAppsSyncException {
        MobileContentYtTable taskOutputInfo = outputConfig.getShopInfo(task.type.getName())
                .orElseThrow(() -> new IllegalStateException("Output config not found for " + task.type.getName()));
        List<YtCluster> outputClusters = mapList(taskOutputInfo.getClusters(), YtCluster::parse);
        YPath outputTableLinkName = YPath.simple(taskOutputInfo.getTable());

        logger.info("Started working on " + task.toString());
        if (!isRecalcNeeded(task, outputClusters, outputTableLinkName)) {
            logger.info("Input tables not updated since the last run, no need to recalc");
            return;
        }

        if (inputTableIsEmpty(task)) {
            logger.warn("At least one input table is empty, exiting");
            return;
        }

        YPath mapped;
        try {
            mapped = runYqlQuery(task);
            logger.info("YQL task complete for " + task.toString());
        } catch (MobileAppsSyncException e) {
            logger.error("Failed to run YQL query for " + task.toString());
            throw e;
        }

        YPath sorted = createSortedTmpTable(tableSchema, task, mapped);
        logger.info("Sorting complete for " + task.toString());

        List<YtCluster> aliveClusters = getAliveClusters(outputClusters);
        if (!aliveClusters.isEmpty()) {
            logger.info("Currently alive output clusters: " + Arrays.toString(mapList(aliveClusters,
                    YtCluster::getName).toArray()));
        } else {
            logger.error("All output clusters down for task " + task.toString());
            throw new MobileAppsSyncException("All output clusters down");
        }

        Map<YtCluster, YPath> transferred = runTransferManager(task, sorted, aliveClusters);
        if (transferred.isEmpty()) {
            throw new MobileAppsSyncException("Failed to transfer table to any cluster");
        }
        logger.info("Transfer completed");

        YPath finalTableName = getOutputTableName(outputTableLinkName);
        transferred.forEach((ytCluster, movedTable) -> {
            try {
                YPath mergedTable = mergeOldData(movedTable, outputTableLinkName, ytCluster, tableSchema);
                replaceOutputTable(task, mergedTable, finalTableName, ytCluster, outputTableLinkName);
            } catch (MobileAppsSyncException e) {
                logger.error("Failed to replace old table on cluster " + ytCluster.getName(), e);
            }
            clearOldData(finalTableName, ytCluster);
        });
        logger.info("Task completed: " + task.toString());
    }

    /**
     * Проверяет что ни одна из входных таблиц не является пустой.
     * Страховка на случай поломки робота в авроре, чтоб не убить разом все приложения
     */
    private boolean inputTableIsEmpty(MobileAppsSyncJobConfig.Task task) {
        return task.inputTables.stream()
                .map(table -> table.attribute(ROW_COUNT_ATTRIBUTE))
                .map(rowCountAttr -> ytProvider.get(task.inputCluster).cypress().get(rowCountAttr).longValue())
                .anyMatch(rowsCount -> rowsCount == 0);
    }

    /**
     * Проверка нужно ли проводить перерасчет для данной задачи.
     * Сравнивает максимальный timestamp входных таблиц с максимальным timestamp выходных.
     * <p>
     * Если вход не найден - бросает {@link MobileAppsSyncException},
     * если выход не найден - true,
     * иначе возвращает input_ts > output_ts
     */
    boolean isRecalcNeeded(MobileAppsSyncJobConfig.Task task,
                           Collection<YtCluster> outputClusters, YPath outputTable) throws MobileAppsSyncException {
        List<YtCluster> aliveClusters = getAliveClusters(outputClusters);

        Instant maxInputTable;
        try {
            maxInputTable = task.inputTables.stream()
                    .map(table -> table.attribute(MODIFICATION_TIME_ATTRIBUTE))
                    .map(timeStr -> ytProvider.get(task.inputCluster).cypress().get(timeStr).stringValue())
                    .map(Instant::parse)
                    .reduce(BinaryOperator.maxBy(Instant::compareTo))
                    .orElseThrow(() -> new MobileAppsSyncException("Unable to check input table timestamps"));
        } catch (Exception e) {
            if (InterruptedException.class.isAssignableFrom(e.getClass())) {
                Thread.currentThread().interrupt();
            }
            throw new MobileAppsSyncException("Unable to check input table timestamps", e);
        }

        Optional<Instant> maxOutputTable;
        try {
            maxOutputTable = aliveClusters.stream()
                    .map(ytProvider::get)
                    .map(Yt::cypress)
                    .filter(c -> c.exists(outputTable))
                    .map(c -> c.get(outputTable.attribute(CREATION_TIME_ATTRIBUTE)).stringValue())
                    .map(Instant::parse)
                    .reduce(BinaryOperator.maxBy(Instant::compareTo));
        } catch (Exception e) {
            if (InterruptedException.class.isAssignableFrom(e.getClass())) {
                Thread.currentThread().interrupt();
            }
            logger.warn("Exception while trying to parse output table timestamps", e);
            return true;
        }

        return maxOutputTable.map(output -> output.isBefore(maxInputTable)).orElse(true);
    }

    /**
     * Проверяет жизнеспособность кластера попыткой получить листинг его корня
     */
    private List<YtCluster> getAliveClusters(Collection<YtCluster> outputClusters) {
        List<YtCluster> aliveClusters = new ArrayList<>();
        for (YtCluster cluster : outputClusters) {
            try {
                ytProvider.get(cluster).cypress().list(YPath.cypressRoot());
                aliveClusters.add(cluster);
            } catch (Exception e) {
                logger.warn("Cluster failed: " + cluster.getName());
            }
        }
        return aliveClusters;
    }

    /**
     * Запускает новый таск по переносу таблицы на ТМ и ждет его выполнения
     */
    private Map<YtCluster, YPath> runTransferManager(MobileAppsSyncJobConfig.Task task, YPath input,
                                                     List<YtCluster> outputClusters) throws MobileAppsSyncException {
        YPath output = YPath.simple(YtPathUtil.generateTemporaryPath());
        Map<String, YtCluster> clustersByTask = listToMap(outputClusters,
                cluster -> {
                    TransferManagerJobConfig tmJob = new TransferManagerJobConfig()
                            .withInputCluster(task.inputCluster.getName())
                            .withOutputCluster(cluster.getName())
                            .withInputTable(input.toString())
                            .withOutputTable(output.toString());
                    return transferManager.queryTransferManager(tmJob);
                },
                identity());

        try {
            Map<String, Boolean> result = transferManager
                    .await(new ArrayList<>(clustersByTask.keySet()), TM_MAX_AWAIT_DURATION);
            return EntryStream.of(clustersByTask)
                    .filterKeys(taskId -> result.getOrDefault(taskId, false))
                    .values()
                    .toMap(ytCluster -> output);
        } catch (TransferManagerException e) {
            throw new MobileAppsSyncException(e);
        } finally {
            ytProvider.get(task.inputCluster).cypress().remove(input);
        }
    }

    /**
     * Вспомогательный метод, возвращающий {@link YPath} к таблице в той же директории, что и предоставленный путь,
     * с именем, равным текущему timestamp
     */
    private YPath getOutputTableName(YPath path) {
        return path.parent().child(Long.toString(Instant.now().getEpochSecond()));
    }

    /**
     * Подставляет в новую таблицу недостающие данные из старой таблицы по ключам.
     * Если такой ключ в новой таблице уже есть - данные не заменяются.
     * Порядок таблиц важен - сначала новая, потом старая.
     *
     * @param newTable    Новая таблица
     * @param oldTable    Старая таблица
     * @param cluster     Кластер
     * @param tableSchema Схема таблицы
     * @return Путь ко временной таблице с результатом
     */
    private YPath mergeOldData(YPath newTable, YPath oldTable, YtCluster cluster,
                               TableSchema tableSchema) {
        logger.info("Merging the new table with the old one");
        Yt yt = ytProvider.get(cluster);
        if (!yt.cypress().exists(oldTable)) {
            logger.info("Nothing to merge");
            return newTable;
        }

        YPath output = YPath.simple(YtPathUtil.generateTemporaryPath());
        yt.cypress().create(output, CypressNodeType.TABLE,
                Map.ofEntries(
                        entry(YtUtils.SCHEMA_ATTR, tableSchema.toYTree()),
                        entry(LAST_UPDATE_TIME_ATTR, nowDateTimeToStringTreeNode())
                ));

        YPath reducerScript = YPath.simple(YtPathUtil.generateTemporaryPath());
        yt.files().write(reducerScript, this.getClass().getResourceAsStream("/mobileappssync/merge_script.py"));

        // Включение автоматической конверсии типов для выходной таблицы
        // Используется для конверсии целых чисел в double где это нужно для обхода особенностей jq
        YTableEntryType<JsonNode> outputFormat = new JacksonTableEntryType();
        outputFormat.format().getAttributes()
                .put("enable_type_conversion", new YTreeBooleanNodeImpl(true, Cf.map()));
        CommandSpec commandSpec = CommandSpec.builder()
                .setCommand("python2 " + reducerScript.name())
                .setInputType(YTableEntryTypes.JACKSON)
                .setOutputType(outputFormat)
                .setFiles(Collections.singletonList(reducerScript))
                .setMemoryLimit(DataSize.fromGigaBytes(2))
                .build();
        ListF<YPath> inputTables = Cf.list(newTable, oldTable);
        ListF<YPath> outputTables = Cf.list(output);
        ListF<String> reduceBy = Cf.list("app_id", "lang");
        ReduceSpec spec = ReduceSpec.builder()
                .setInputTables(inputTables)
                .setOutputTables(outputTables)
                .setReduceBy(reduceBy)
                .setReducerSpec(commandSpec)
                .setJobIo(JobIo.builder()
                        .setEnableTableIndex(false)
                        .setEnableRowIndex(false)
                        .build())
                .setAdditionalSpecParameters(Cf.map("title",
                        new YTreeStringNodeImpl("MobileAppsSyncJob: merge old data", Cf.map())))
                .build();

        try {
            yt.operations()
                    .reduceAndGetOp(spec)
                    .awaitAndThrowIfNotSuccess(MAX_SORT_DURATION);
        } finally {
            ytProvider.get(cluster).cypress().remove(newTable);
        }
        return output;
    }

    private YTreeStringNode nowDateTimeToStringTreeNode() {
        return new YTreeStringNodeImpl(ZonedDateTime.now().toString(), emptyMap());
    }

    /**
     * Подменяет ссылку на старую динамическую таблицу новой.
     * <p>
     * Структура директории:<br>
     * - parent<br>
     * -- latest -> $(timestamp)<br>
     * -- $(timestamp)<br>
     * -- $(older-timestamp)<br>
     * -- ...
     */
    private void replaceOutputTable(MobileAppsSyncJobConfig.Task task, YPath input, YPath outputTableName,
                                    YtCluster cluster, YPath outputTableLinkName) throws MobileAppsSyncException {
        logger.info("Creating new table named " + outputTableName.name());
        Yt yt = ytProvider.get(cluster);
        Transaction transaction = yt.transactions().startAndGet(MAX_REPLACE_TOTAL_DURATION);
        try {
            yt.cypress().move(input, outputTableName);
            changeYtTableMedium(task, outputTableName, yt);
            yt.tables().alterTable(outputTableName, Optional.of(true), Optional.empty());
            yt.tables().mount(outputTableName);
            waitForNodeMountState(yt, outputTableName, WAIT_MOUNT_DURATION, YtNodeMountState.MOUNTED);
            yt.cypress().link(outputTableName, outputTableLinkName, /* force = */ true);
            transaction.commit();
        } catch (Exception e) {
            logger.error("Failed to replace old output table with new one");
            transaction.abort();
            throw new MobileAppsSyncException(e);
        } finally {
            if (yt.cypress().exists(input)) {
                yt.cypress().remove(input);
            }
        }
    }

    /**
     * Меняет тип хранилища (медиум) на указанный
     */
    private void changeYtTableMedium(MobileAppsSyncJobConfig.Task task, YPath latest, Yt yt) {
        YTreeMapNode newMediaProperties = new YTreeMapNodeImpl(Cf.map());
        newMediaProperties.put("data_parts_only", new YTreeBooleanNodeImpl(false, Cf.map()));
        newMediaProperties.put("replication_factor", new YTreeIntegerNodeImpl(/* signed = */true, 3, Cf.map()));

        YTreeMapNode currentMedia = yt.cypress().get(latest.child("@media")).mapNode();
        currentMedia.put(task.mediaType, newMediaProperties);
        yt.cypress().set(latest.child("@media"), currentMedia);
        yt.cypress().set(latest.child("@primary_medium"), new YTreeStringNodeImpl(task.mediaType, Cf.map()));

        YTreeMapNode newMedia = new YTreeMapNodeImpl(Cf.map());
        newMedia.put(task.mediaType, newMediaProperties);
        yt.cypress().set(latest.child("@media"), newMedia);
    }

    /**
     * Ожидает изменения статуса монтирования ноды в течение указанного времени
     */
    void waitForNodeMountState(Yt yt, YPath path, Duration maxDuration, YtNodeMountState targetState)
            throws TimeoutException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        while (stopwatch.elapsed(TimeUnit.SECONDS) < maxDuration.getSeconds()) {
            if (mountStateEquals(yt, path, targetState)) {
                return;
            }
            ThreadUtils.sleep(Duration.ofSeconds(1));
        }
        throw new TimeoutException("Unable to mount/unmount table: " + path.toString());
    }

    /**
     * Возвращает true если статус монтирования ноды по указанному пути равен targetState
     */
    boolean mountStateEquals(Yt yt, YPath path, YtNodeMountState targetState) {
        return targetState.name.equals(yt.cypress().get(path.child("@tablet_state")).stringValue());
    }

    /**
     * Удаляет старые данные.
     *
     * @param latest        Последняя созданная выходная таблица
     * @param outputCluster Кластер
     */
    void clearOldData(YPath latest, YtCluster outputCluster) {
        Yt yt = ytProvider.get(outputCluster);
        Cypress cypress = yt.cypress();
        YPath directory = latest.parent();
        List<YPath> nodesWithLinks = cypress.list(directory)
                .stream()
                .map(s -> directory.child(s.getValue() + LINK_TARGET_PATH))
                .filter(cypress::exists)
                .map(node -> YPath.simple(cypress.get(node).stringValue()))
                .collect(Collectors.toList());
        List<YPath> nodesToDelete = Cf.wrap(cypress.list(directory))
                .filter(node -> StringUtils.isNumeric(node.getValue()))
                .map(node -> directory.child(node.getValue()))
                .filterNot(nodesWithLinks::contains)
                .filterNot(node -> node.equals(latest));


        for (YPath node : nodesToDelete) {
            try {
                if (!mountStateEquals(yt, node, YtNodeMountState.UNMOUNTED)) {
                    yt.tables().unmount(node);
                    waitForNodeMountState(yt, node, WAIT_MOUNT_DURATION, YtNodeMountState.UNMOUNTED);
                }
            } catch (YtException e) {
                logger.error("Error while unmounting old table " + node.name(), e);
            } catch (TimeoutException e) {
                logger.error("Timeout while unmounting old table", e);
            }

            try {
                cypress.remove(node);
            } catch (YtException e) {
                logger.error("Error while removing old table " + node.name(), e);
            }
        }
    }

    /**
     * Создает упорядоченную статическую таблицу с указанной схемой и переносит в нее результат YQL маппера.
     *
     * @param tableSchema Схема таблицы
     * @param task        Конфигурация переноса
     * @param input       Входная таблица
     */
    private YPath createSortedTmpTable(TableSchema tableSchema, MobileAppsSyncJobConfig.Task task, YPath input)
            throws MobileAppsSyncException {
        Yt yt = ytProvider.get(task.inputCluster);
        YPath output = YPath.simple(YtPathUtil.generateTemporaryPath());

        if (yt.cypress().exists(output)) {
            yt.cypress().remove(output);
        }
        yt.cypress().create(output, CypressNodeType.TABLE, Cf.map(YtUtils.SCHEMA_ATTR, tableSchema.toYTree()));
        Operation op = yt.operations().sortAndGetOp(input, output,
                Cf.x(tableSchema.toKeys().getColumns()).map(ColumnSchema::getName));
        try {
            op.awaitAndThrowIfNotSuccess(MAX_SORT_DURATION);
            logger.info("Sorting completed");
        } catch (YtException e) {
            var message = String.format("Sorting operation %s failed or timed out", op.getId());
            throw new MobileAppsSyncException(message, e);
        } finally {
            yt.cypress().remove(input);
        }

        return output;
    }

    /**
     * Запускает YQL с запросом из указанного в {@link MobileAppsSyncJobConfig.Task} файла, который парсит данные
     * исходной таблицы.
     *
     * @param task Конфигурация переноса
     * @throws MobileAppsSyncException в случае если маппер упал
     */
    private YPath runYqlQuery(MobileAppsSyncJobConfig.Task task) throws MobileAppsSyncException {
        YPath output = YPath.simple(YtPathUtil.generateTemporaryPath());
        YqlDataSource dataSource = ytProvider.getYql(task.inputCluster, YtSQLSyntaxVersion.SQLv1);
        String query = createYqlQuery(task, output);
        try (Connection connection = dataSource.getConnection();
             PreparedStatement statement = connection.prepareStatement(query)) {
            statement.execute();
        } catch (SQLException e) {
            logger.error("Exception while processing " + task.toString(), e);
            throw new MobileAppsSyncException(e);
        }
        logger.info("YQL task completed");

        return output;
    }


    /**
     * Подменяет в шаблоне запроса значение %(tablepath) на реальный путь к таблице.
     *
     * @param task   Конфигурация
     * @param output Выходная таблица
     */
    private String createYqlQuery(MobileAppsSyncJobConfig.Task task, YPath output) throws MobileAppsSyncException {
        String templatePath = Optional.ofNullable(MobileAppsSyncJobConfig.YQL_TMPL_FILES.get(task.type))
                .orElseThrow(() -> new IllegalStateException("Unable to find template for " + task.type.getName()));
        String queryPath = Optional.ofNullable(MobileAppsSyncJobConfig.YQL_FILES.get(task.type))
                .orElseThrow(() -> new IllegalStateException("Unable to find query file for " + task.type.getName()));

        String selectTmpl = new ClassPathResourceInputStreamSource(templatePath).readText();
        List<String> inputs = mapList(task.inputTables, tablePath ->
                new StrSubstitutor(singletonMap("tablepath", tablePath), "%(", ")").replace(selectTmpl));
        String inputUnion = inputs.stream().reduce((a, b) -> String.format("%s\nUNION ALL\n%s", a, b))
                .orElseThrow(() -> new MobileAppsSyncException("Input tables union empty"));

        String query = new ClassPathResourceInputStreamSource(queryPath).readText();
        Map<String, String> values = new HashMap<>();
        values.put("tmppath", output.toString());
        values.put("select_parts", inputUnion);
        StrSubstitutor sub = new StrSubstitutor(values, "%(", ")");
        return sub.replace(query);
    }
}
