package ru.yandex.direct.jobs.mobileappssync;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nonnull;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.common.mobilecontent.MobileContentStoreType;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.direct.common.mobilecontent.MobileContentStoreType.GOOGLE_PLAY;
import static ru.yandex.direct.common.mobilecontent.MobileContentStoreType.ITUNES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Работа с конфигурацией для {@link MobileAppsSyncJob}.
 * Пример конфига:
 * <p>
 * mobile_apps_data: {<br>
 * &nbsp;gplay:{<br>
 * &nbsp;&nbsp;input_cluster: "hahn"<br>
 * &nbsp;&nbsp;input_tables: ["//home/extdata/mobile/google_play/latest/ru", "//home/extdata/mobile/google_play/latest/ua"...]<br>
 * &nbsp;&nbsp;media_type: "ssd_blobs"<br>
 * &nbsp;}<br>
 * &nbsp;itunes:{<br>
 * &nbsp;&nbsp;.......
 * &nbsp;}
 * }<br>
 * <br>
 * transfer_manager: {<br>
 * &nbsp;transfer_manager_api: "http://transfer-manager.yt.yandex.net/api/v1/tasks/"<br>
 * &nbsp;token: "~/.yt/token"<br>
 * &nbsp;max_retries: 3<br>
 * }<br>
 * <p>
 * <p>
 * Поле lang (язык) берется из последней части имени таблицы.
 * То есть для "//home/extdata/mobile/google_play/latest/ru" оно будет равно "ru"
 * <p>
 * Выходные таблицы берутся из common configuration, так как они должны быть синхронизированы с web/intapi.
 */
public class MobileAppsSyncJobConfig {
    static final Map<MobileContentStoreType, String> YQL_FILES = ImmutableMap.of(
            GOOGLE_PLAY, "mobileappssync/gplay.yql",
            ITUNES, "mobileappssync/itunes.yql"
    );
    static final Map<MobileContentStoreType, String> YQL_TMPL_FILES = ImmutableMap.of(
            GOOGLE_PLAY, "mobileappssync/gplay_select_part_tmpl.yql",
            ITUNES, "mobileappssync/itunes_select_part_tmpl.yql"
    );
    private final List<Task> tasksToProcess;

    public MobileAppsSyncJobConfig(DirectConfig directConfig) {
        DirectConfig mobileAppsData = directConfig.getBranch("mobile_apps_data");
        this.tasksToProcess = new ArrayList<>();
        for (MobileContentStoreType type : MobileContentStoreType.values()) {
            Optional<DirectConfig> branch = mobileAppsData.findBranch(type.getName());
            branch.ifPresent(conf -> tasksToProcess.add(new Task(conf, type.getName())));
        }

    }

    public List<Task> getTasks() {
        return Collections.unmodifiableList(tasksToProcess);
    }

    /**
     * Описание одной подзадачи (один магазин, одна выходная таблица)
     */
    public class Task {
        @Nonnull
        public final YtCluster inputCluster;
        @Nonnull
        public final List<YPath> inputTables;
        @Nonnull
        public final MobileContentStoreType type;
        @Nonnull
        public final String mediaType;

        Task(DirectConfig config, String type) {
            this.inputCluster = YtCluster.valueOf(config.getString("input_cluster").toUpperCase());
            this.inputTables = mapList(config.getStringList("input_tables"), YPath::simple);
            this.type = MobileContentStoreType.byName(type);
            this.mediaType = config.getString("media_type");
        }

        @Override
        public String toString() {
            return "Task{" +
                    "inputCluster=" + inputCluster +
                    ", inputTables=" + Arrays.toString(mapList(inputTables, YPath::toString).toArray()) +
                    ", type=" + type +
                    '}';
        }
    }
}
