package ru.yandex.direct.jobs.mobileappsverification;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.validation.constraints.NotNull;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.appmetrika.AppMetrikaClient;
import ru.yandex.direct.appmetrika.model.Platform;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppConversionStatisticRepository;
import ru.yandex.direct.core.entity.mobileapp.repository.MobileAppRepository;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.dbschema.ppc.enums.MobileContentOsType;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.juggler.check.model.NotificationRecipient;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.DirectShardedJob;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Джоба синхронизирует флаги верификации мобильных приложений,
 * источник - динамическая таблица //home/yabs/stat/DirectMobileAppStat,
 * место назначения - таблица mobile_apps в MySql.
 * Также джоба ходит в апи AppMetrika и верифицирует приложения, добавленнные там.
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(hours = 2),
        tags = {DIRECT_PRIORITY_1},
        notifications = {
                @OnChangeNotification(recipient = NotificationRecipient.LOGIN_IVATKOEGOR,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        needCheck = ProductionOnly.class)
@Hourglass(cronExpression = "0 0 * * * ?")
@ParametersAreNonnullByDefault
public class MobileAppsVerificationJob extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(MobileAppsVerificationJob.class);

    private final MobileAppConversionStatisticRepository mobileAppConversionStatisticRepository;
    private final MobileAppRepository mobileAppRepository;
    private final MobileContentRepository mobileContentRepository;
    private final ClientService clientService;
    private final AppMetrikaClient appMetrikaClient;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    private static final long CHUNK_SIZE = 500L;

    @Autowired
    public MobileAppsVerificationJob(
            MobileAppConversionStatisticRepository mobileAppConversionStatisticRepository,
            MobileAppRepository mobileAppRepository,
            MobileContentRepository mobileContentRepository,
            ClientService clientService,
            AppMetrikaClient appMetrikaClient,
            PpcPropertiesSupport ppcPropertiesSupport
    ) {
        this.mobileAppConversionStatisticRepository = mobileAppConversionStatisticRepository;
        this.mobileAppRepository = mobileAppRepository;
        this.mobileContentRepository = mobileContentRepository;
        this.clientService = clientService;
        this.appMetrikaClient = appMetrikaClient;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void execute() {
        long lastId = 0;
        while (true) {
            var mobileApps = mobileAppRepository.getMobileAppsChunk(getShard(), lastId, CHUNK_SIZE);
            if (mobileApps.isEmpty()) {
                break;
            }

            lastId = mobileApps.get(mobileApps.size() - 1).getId();
            verifyApps(mobileApps);
            verifyAppsFromSameBundle(mobileApps);
        }
    }

    private void verifyAppsFromSameBundle(List<MobileApp> mobileApps) {
        var appIds = mapList(mobileApps, MobileApp::getId);
        Arrays.stream(MobileContentOsType.values()).forEach(osType -> {
            var storeAppIdWithClientIds = mobileAppRepository.getStoreAppIdsToClientsWithVerification(
                    getShard(),
                    appIds,
                    osType
            );
            mobileAppRepository.updateVerificationByStoreAppId(
                    getShard(),
                    storeAppIdWithClientIds,
                    true,
                    osType
            );
        });
    }

    private void verifyApps(List<MobileApp> mobileApps) {
        fillMobileContent(mobileApps);
        var apps = mobileApps.stream()
                .filter(app -> app.getMobileContent() != null)
                .collect(Collectors.groupingBy(it -> it.getMobileContent().getOsType()));
        apps.forEach((osType, osTypeApps) -> verifyApps(osTypeApps, osType));
    }

    private void verifyApps(List<MobileApp> apps, OsType osType) {
        var externalVerifiedAppIds = getExternalTrackersVerification(apps);
        var appMetricaVerifiedAppIds = getAppMetrikaVerification(apps, osType);

        var newVerifiedAppIds = apps.stream()
                .filter(mobileApp ->
                        (mobileApp.getHasVerification() == null || !mobileApp.getHasVerification())
                                && (externalVerifiedAppIds.contains(mobileApp.getId())
                                || appMetricaVerifiedAppIds.contains(mobileApp.getId())))
                .map(MobileApp::getId)
                .collect(toList());

        logger.info("Updating {} apps with verification", newVerifiedAppIds);
        mobileAppRepository.updateVerification(getShard(), newVerifiedAppIds, true);
    }

    @NotNull
    private Set<Long> getExternalTrackersVerification(List<MobileApp> apps) {
        Set<String> storeAppId = mapAndFilterToSet(
                apps,
                this::getStoreAppId,
                appId -> !Strings.isNullOrEmpty(appId)
        );
        var mobileAppIds = listToSet(apps, MobileApp::getId);
        return new HashSet<>(mobileAppConversionStatisticRepository.getVerifiedAppIds(storeAppId, mobileAppIds));
    }

    private HashSet<Long> getAppMetrikaVerification(List<MobileApp> apps, OsType osType) {
        var clientIds = mapList(apps, MobileApp::getClientId);
        var clients = clientService.getClients(getShard(), clientIds);
        var uidsForClientIds = listToMap(clients, Client::getClientId, Client::getChiefUid);

        Map<UidWithStoreAppId, List<Long>> uidsWithStoreAppIds = apps.stream()
                .filter(app -> !Strings.isNullOrEmpty(getStoreAppId(app))
                        && uidsForClientIds.containsKey(app.getClientId().asLong()))
                .collect(groupingBy(
                        app -> new UidWithStoreAppId(
                                uidsForClientIds.get(app.getClientId().asLong()), getStoreAppId(app)
                        ),
                        Collectors.mapping(MobileApp::getId, toList())));

        var verifiedAppIds = new HashSet<Long>();
        try {
            var platform = getPlatform(osType);
            long iteration = 0L;
            long requestLimit = getAppMetricaRequestLimit();
            long idleTime = getAppMetricaIdleTimeInMills();

            for (var entry : uidsWithStoreAppIds.entrySet()) {
                var applications = appMetrikaClient.getApplications(
                        entry.getKey().getUid(), entry.getKey().getStoreAppId(), platform, null, null, null);
                if (applications != null && !applications.isEmpty()) {
                    // verify all apps that belong to particular user and bundle
                    verifiedAppIds.addAll(entry.getValue());
                }
                if (++iteration >= requestLimit) {
                    Thread.sleep(idleTime);
                    iteration = 0L;
                }
            }
        } catch (Exception ex) {
            logger.error("Error while requesting AppMetrika", ex);
        }

        return verifiedAppIds;
    }

    private Platform getPlatform(OsType osType) {
        switch (osType) {
            case ANDROID:
                return Platform.android;
            case IOS:
                return Platform.ios;
            default:
                throw new IllegalStateException("Unknown osType: " + osType);
        }
    }

    private void fillMobileContent(List<MobileApp> mobileApps) {
        Map<Long, MobileContent> mobileContentById = mobileContentRepository
                .getMobileContent(getShard(), mapList(mobileApps, MobileApp::getMobileContentId))
                .stream()
                .collect(toMap(MobileContent::getId, identity()));

        for (var mobileApp : mobileApps) {
            mobileApp.setMobileContent(mobileContentById.get(mobileApp.getMobileContentId()));
        }
    }

    @Nullable
    private String getStoreAppId(MobileApp mobileApp) {
        if (mobileApp.getMobileContent() == null) {
            return null;
        }

        if (mobileApp.getMobileContent().getOsType().equals(OsType.ANDROID)) {
            return mobileApp.getMobileContent().getStoreContentId();
        } else {
            return mobileApp.getMobileContent().getBundleId();
        }
    }

    private Long getAppMetricaIdleTimeInMills() {
        return ppcPropertiesSupport
                .get(PpcPropertyNames.APP_METRICA_REQUEST_IDLE_TIME, Duration.ofMinutes(1))
                .getOrDefault(1000L);
    }

    private Long getAppMetricaRequestLimit() {
        return ppcPropertiesSupport
                .get(PpcPropertyNames.APP_METRICA_REQUEST_LIMIT, Duration.ofMinutes(1))
                .getOrDefault(120L);
    }
}
