package ru.yandex.direct.jobs.mobilegoals

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod.TELEGRAM
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.MOBILE_APPS_AUTO_GOALS_JOB_ENABLED
import ru.yandex.direct.core.entity.mobilegoals.MobileAppAutoGoalsService
import ru.yandex.direct.core.entity.mobilegoals.MobileAppGoalsExternalTrackerRepository
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.env.TypicalEnvironment
import ru.yandex.direct.juggler.JugglerStatus.*
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.JugglerCheck.Duration
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_NBAHOB
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectShardedJob

/**
 * Джоба автодобавления целей для мобильных приложений на основе статистики конверсий DirectMobileAppStat
 */
@JugglerCheck(
    ttl = Duration(hours = 7),
    needCheck = ProductionOnly::class,
    tags = [CheckTag.YT, CheckTag.DIRECT_PRIORITY_1],
    notifications = [OnChangeNotification(
        recipient = [LOGIN_NBAHOB],
        method = [TELEGRAM],
        status = [OK, WARN, CRIT]
    )]
)
@Hourglass(periodInSeconds = 3 * 60 * 60, needSchedule = TypicalEnvironment::class)
class MobileAppAutoGoalsJob @Autowired constructor(
    private val ppcPropertiesSupport: PpcPropertiesSupport,
    private val mobileAppAutoGoalsService: MobileAppAutoGoalsService,
    private val mobileAppGoalsExternalTrackerRepository: MobileAppGoalsExternalTrackerRepository,
) : DirectShardedJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(MobileAppAutoGoalsJob::class.java)
        private const val DAYS_CHECKED = 1
    }

    override fun execute() {
        val jobEnabled = ppcPropertiesSupport.get(MOBILE_APPS_AUTO_GOALS_JOB_ENABLED).getOrDefault(false)
        if (!jobEnabled) {
            logger.info("Skip processing, job is not enabled")
            return
        }
        val eventsToAdd = mobileAppAutoGoalsService.getAutoGoalsToAdd(shard, DAYS_CHECKED)
        if (eventsToAdd.isEmpty()) {
            logger.info("No goals added")
            return
        }
        mobileAppGoalsExternalTrackerRepository.addEvents(shard, eventsToAdd)
        logger.info("Added ${eventsToAdd.size} goals")
    }
}
