package ru.yandex.direct.jobs.moderation;

import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl.LogbrokerBatchReaderImpl;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.jobs.moderation.config.LogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.config.TopicWithGroup;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Базовая джоба обработки запросов модерации
 * Читает запросы из {@code getConsumerProperties()}
 * Также инициализирует различные метрики
 */
public abstract class BaseReceiveModerationJob<T extends AbstractModerationResponse> extends DirectParameterizedJob {
    private static final Logger logger = LoggerFactory.getLogger(BaseReceiveModerationJob.class);

    protected final ResponseModerationParameters moderationParameters;
    protected final ModerationResponseProcessorFilter processorFilter;

    private final LogbrokerConsumerPropertiesHolder logbrokerConsumerPropertiesHolder;
    private final ReceiveModerationService receiveModerationService;

    private boolean isInitialized = false;
    private Labels solomonRegistryLabels;
    private MetricRegistry metricRegistry;
    private RetryingLogbrokerBatchReader<T> logbrokerReader;

    public BaseReceiveModerationJob(ResponseModerationParameters moderationParameters,
                                    ModerationResponseProcessorFilter processorFilter,
                                    LogbrokerConsumerPropertiesHolder logbrokerConsumerPropertiesHolder,
                                    ReceiveModerationService receiveModerationService) {
        this.moderationParameters = moderationParameters;
        this.logbrokerConsumerPropertiesHolder = logbrokerConsumerPropertiesHolder;
        this.receiveModerationService = receiveModerationService;
        this.processorFilter = processorFilter;
    }

    private void initializeInternal() {
        if (!isInitialized) {
            TopicWithGroup param = logbrokerConsumerPropertiesHolder.convertStringToParam(getParam());
            LogbrokerConsumerProperties consumerProperties =
                    logbrokerConsumerPropertiesHolder.getPropertiesByParam(param);

            solomonRegistryLabels = Labels.of("jobs", "moderation_reader", "topic", consumerProperties.getReadTopic());

            metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);

            var syncConsumerSupplier = receiveModerationService.createConsumerSupplier(consumerProperties);
            initializeBeforeReader();
            logbrokerReader = new RetryingLogbrokerBatchReader<>(
                    () -> getLogbrokerBatchReader(syncConsumerSupplier),
                    consumerProperties.getRetries());
            isInitialized = true;
        }
    }

    /**
     * Возвращает {@link MetricRegistry} на конкретный топик логброкера
     */
    protected MetricRegistry getMetricRegistry() {
        checkState(metricRegistry != null, "metricRegistry not initialized");
        return metricRegistry;
    }

    /**
     * Инициализация перед выполнением
     */
    protected void initializeBeforeReader() {
    }

    /**
     * Конвертер приходящего потока событий
     */
    protected abstract LogbrokerBatchReaderImpl<T> getLogbrokerBatchReader(Supplier<SyncConsumer> syncConsumerSupplier);

    /**
     * Обработчик запросов
     */
    protected abstract BaseModerationResponseProcessor<T> getEventsProcessor();

    @Override
    public void execute() {
        initializeInternal();
        try {
            logbrokerReader.fetchEvents(getEventsProcessor());
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    @Override
    public void finish() {
        if (solomonRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels);
        }
        if (logbrokerReader != null) {
            try {
                this.logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
        isInitialized = false;
    }
}
