package ru.yandex.direct.jobs.moderation;

import java.util.List;
import java.util.function.BiConsumer;

import ru.yandex.direct.common.log.container.ModerationLogEntry;
import ru.yandex.direct.common.log.service.ModerationLogService;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationError;
import ru.yandex.direct.jobs.moderation.errors.ModerationErrorCodeToJugglerStatus;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.juggler.JugglerStatus.OK;

class ModerationErrorResponseProcessor extends BaseModerationResponseProcessor<ModerationError> {
    static final String ERROR_RESPONSE = "error_response";

    private final Counter errorResponseCount;
    private final ModerationLogService moderationLogService;
    private final BiConsumer<JugglerStatus, String> jugglerDelegate;
    private final ModerationErrorCodeToJugglerStatus moderationErrorCodeToJugglerStatus;

    ModerationErrorResponseProcessor(ModerationResponseProcessorFilter filter, MetricRegistry metricRegistry,
                                     ModerationLogService moderationLogService,
                                     BiConsumer<JugglerStatus, String> jugglerDelegate,
                                     ModerationErrorCodeToJugglerStatus moderationErrorCodeToJugglerStatus) {
        super(filter);
        this.errorResponseCount = metricRegistry.counter(ERROR_RESPONSE);
        this.moderationLogService = moderationLogService;
        this.jugglerDelegate = jugglerDelegate;
        this.moderationErrorCodeToJugglerStatus = moderationErrorCodeToJugglerStatus;
    }

    @Override
    protected void processResponses(List<ModerationError> responses) {
        if (responses.isEmpty()) {
            jugglerDelegate.accept(OK, "No new error responses");
            return;
        }

        JugglerStatus status = null;
        int jugglerStatusWeight = Integer.MIN_VALUE;

        for (ModerationError error : responses) {
            BannerModerationMeta meta = error.getMeta();
            moderationLogService.logEvent(new ModerationLogEntry<>(meta.getCampaignId(), meta.getAdGroupId(),
                    meta.getBannerId(), ModerationLogEntry.Action.RESPONSE, ModerationLogEntry.Source.ERROR,
                    false, error));

            JugglerStatus newStatus = moderationErrorCodeToJugglerStatus.getJugglerStatus(error.getCode());
            int newWeight = jugglerStatusWeight(newStatus);

            if (newStatus != OK && newWeight > jugglerStatusWeight) {
                jugglerStatusWeight = newWeight;
                status = newStatus;
            }

        }

        errorResponseCount.add(responses.size());

        if (status != null) {
            jugglerDelegate.accept(status, "moderation return error response");
        }
    }

    int jugglerStatusWeight(JugglerStatus status) {
        switch (status) {
            case OK:
                return 0;
            case INFO:
                return 1;
            case WARN:
                return 2;
            case CRIT:
                return 3;
            default:
                throw new IllegalArgumentException("Unknown juggler status " + status);
        }
    }

}
