package ru.yandex.direct.jobs.moderation;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.moderation.repository.sending.ModerationSendingRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.DirectShardedJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 150),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_2})
@Hourglass(periodInSeconds = 60 * 10)
public class ModerationReadyObjectsMonitoring extends DirectShardedJob {
    private static final Logger logger = LoggerFactory.getLogger(ModerationReadyObjectsMonitoring.class);

    private final List<ModerationSendingRepository> moderationSendingRepositories;
    private MetricRegistry metricRegistry;
    private final Labels solomonRegistryLabels;
    private final DslContextProvider dslContextProvider;
    private final SolomonPushClient solomonPushClient;

    @Autowired
    public ModerationReadyObjectsMonitoring(List<ModerationSendingRepository> moderationSendingRepositories,
                                            DslContextProvider dslContextProvider,
                                            SolomonPushClient solomonPushClient) {
        this.moderationSendingRepositories = moderationSendingRepositories;
        this.dslContextProvider = dslContextProvider;
        this.solomonPushClient = solomonPushClient;
        solomonRegistryLabels = Labels.of("jobs", "moderation_ready_objects_count");
    }

    @Override
    public void initialize(TaskParametersMap taskParametersMap) {
        super.initialize(taskParametersMap);
        metricRegistry = SolomonUtils.newPushRegistry(solomonRegistryLabels);
    }

    @Override
    public void execute() {
        long readyCount = 0;

        for (var repository : moderationSendingRepositories) {
            long partSum = repository.getReadyObjectsCount(dslContextProvider.ppc(getShard()).configuration());
            readyCount += partSum;

            metricRegistry.gaugeInt64(repository.getClass().getSimpleName(), Labels.of("shard",
                    String.valueOf(getShard()))).add(partSum);
        }

        metricRegistry.gaugeInt64("total", Labels.of("shard", String.valueOf(getShard()))).add(readyCount);
    }

    @Override
    public void finish() {
        super.finish();
        solomonPushClient.sendMetrics(metricRegistry);
    }
}
