package ru.yandex.direct.jobs.moderation;


import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.text.StrSubstitutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;
import ru.yandex.direct.scheduler.support.DirectJob;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_2;

/**
 * Джоба которая проверяет количество объектов на которые не пришло ответа из модерации за последние 48 часов
 * <p>
 * В случае нахождения таких объектов надо проверить по логам:
 * - что эти объекты действительно требуют ответа (там могут быть нюансы, как с "операторами", если это так то
 * надо поправить SQL в мониторинге - moderation/unanswered_responses.sql)
 * - если требовался ответ, то надо обратиться к команде модерации чтобы узнать что происходит
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 210),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_MODERATION, DIRECT_PRIORITY_2})
@Hourglass(periodInSeconds = 3600)
public class ModerationResponsesTimeMonitoringJob extends DirectJob {
    private final DatabaseWrapperProvider dbProvider;
    private static final String SQL_FILE_PATH = "moderation/unanswered_responses.sql";
    private static final Logger logger = LoggerFactory.getLogger(ModerationResponsesTimeMonitoringJob.class);
    private final Labels solomonRegistryLabels;
    private MetricRegistry metricRegistry;
    private final SolomonPushClient solomonPushClient;

    private static final int EXAMPLES_COUNT = 20;
    private static final int SLICES = 100;

    @Autowired
    public ModerationResponsesTimeMonitoringJob(DatabaseWrapperProvider dbProvider,
                                                SolomonPushClient solomonPushClient) {
        this.dbProvider = dbProvider;
        this.solomonPushClient = solomonPushClient;
        solomonRegistryLabels = Labels.of("jobs", "moderation_responses_monitoring");
    }

    @Override
    public void initialize(TaskParametersMap taskParametersMap) {
        super.initialize(taskParametersMap);
        metricRegistry = SolomonUtils.newPushRegistry(solomonRegistryLabels);
    }

    @Override
    public void execute() {
        String selectTmpl = new ClassPathResourceInputStreamSource(SQL_FILE_PATH).readText();

        //2020-04-08 20:37:01
        LocalDateTime date = LocalDateTime.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        String currentDate = formatter.format(date);

        for (int i = 0; i < SLICES; i++) {
            if (checkLogsSlice(SLICES, i, currentDate, selectTmpl)) {
                break;
            }
        }
    }

    private boolean checkLogsSlice(int slices, int sliceNumber, String date, String selectTmpl) {
        Map<String, String> sqlParams = Map.of(
                "now", date,
                "window", "7",
                "slices_count", Integer.toString(slices),
                "slice_number", Integer.toString(sliceNumber),
                "sla", "8");

        StrSubstitutor sub = new StrSubstitutor(sqlParams, "%(", ")");

        List<Info> infos = dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD).query(sub.replace(selectTmpl),
                (el, n) -> new ModerationResponsesTimeMonitoringJob.Info(
                        el.getLong("bid"),
                        removeSuffix(el.getString("version")),
                        el.getString("type")
                ));

        if (infos.size() > 0) {
            metricRegistry.gaugeInt64("not_answered_moderation_responses").add(infos.size());
            logger.info("Found " + infos.size() + " unanswered requests");

            infos.stream()
                    .limit(EXAMPLES_COUNT)
                    .forEach(e -> logger.info("No answer from moderation: " + e));

            return true;
        } else {
            return false;
        }
    }

    static String removeSuffix(String version) {
        if (version.endsWith("A") || version.endsWith("B")) {
            return version.substring(0, version.length() - 1);
        }
        return version;
    }

    @Override
    public void finish() {
        super.finish();
        solomonPushClient.sendMetrics(metricRegistry);
    }

    public static class Info {
        private final long bid;
        private final String version;
        private final String type;

        public Info(long bid, String version, String type) {
            this.bid = bid;
            this.version = version;
            this.type = type;
        }

        public long getBid() {
            return bid;
        }

        public String getVersion() {
            return version;
        }

        public String getType() {
            return type;
        }

        @Override
        public String toString() {
            return "{" +
                    "bid=" + bid +
                    ", version=" + version +
                    ", type='" + type + '\'' +
                    '}';
        }
    }

}
