package ru.yandex.direct.jobs.moderation;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl.LogbrokerBatchReaderImpl;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.service.ModerationLogService;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentCondition;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessor;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.jobs.moderation.processor.handlers.ModerationResponseHandler;
import ru.yandex.direct.jobs.moderation.reader.ModerationResponseLogbrokerReader;
import ru.yandex.direct.jobs.moderation.reader.support.ModerationResponseConverter;
import ru.yandex.direct.jobs.moderation.reader.support.ModerationResponseSupportFacade;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Обработка вердиктов модерации
 * Для добавления обработки нового итпа вердикта:
 * 1) Описать конвертацию json-а в новом классе extends ModerationResponseConverter
 * 2) Добавить обработчик (implements ModerationResponseHandler)
 * Возможные ошибки:
 * Падения в обработки запросов
 * Не запуск джобы шедулером
 * Проблемы чтения данных из логброкера
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20),
        tags = {DIRECT_PRIORITY_0, CheckTag.JOBS_RELEASE_REGRESSION},
        needCheck = TestingOnly.class
)
@HourglassDaemon(runPeriod = 0)
@Hourglass(periodInSeconds = 0, needSchedule = ReceiveModerationResponseJob.ScheduleCondition.class)
@ParameterizedBy(parametersSource = ResponseModerationLogbrokerConsumerPropertiesHolder.class)
@ParametersAreNonnullByDefault
public class ReceiveModerationResponseJob extends BaseReceiveModerationJob<AbstractModerationResponse> {

    private final List<ModerationResponseConverter> typeSupports;
    private final ShardHelper shardHelper;
    private final List<ModerationResponseHandler> handlers;
    private final ModerationLogService moderationLogService;
    private final PpcPropertiesSupport propertiesSupport;

    private ModerationReadMonitoring monitoring;

    /**
     * Условие на тип окружения, в котором задача должна работать.
     * <p>
     * Должно работать в проде, ТС и devtest
     * на dev7 нет, т.к. консьюмер один и конфликтует с devtest
     */
    @Component
    public static class ScheduleCondition extends EnvironmentCondition {
        @Autowired
        public ScheduleCondition(EnvironmentTypeProvider environmentTypeProvider) {
            super(environmentTypeProvider, EnvironmentType.PRODUCTION, EnvironmentType.TESTING,
                    EnvironmentType.DEVTEST, EnvironmentType.DEVELOPMENT);
        }
    }

    @Autowired
    public ReceiveModerationResponseJob(ResponseModerationParameters moderationParameters,
                                        ReceiveModerationService receiveModerationService,
                                        ModerationResponseProcessorFilter processorFilter,
                                        ResponseModerationLogbrokerConsumerPropertiesHolder moderationLogbrokerConsumerPropertiesHolder,
                                        List<ModerationResponseConverter> typeSupports,
                                        ShardHelper shardHelper,
                                        List<ModerationResponseHandler> handlers,
                                        ModerationLogService moderationLogService,
                                        PpcPropertiesSupport propertiesSupport) {
        super(moderationParameters, processorFilter, moderationLogbrokerConsumerPropertiesHolder,
                receiveModerationService);
        this.typeSupports = typeSupports;
        this.shardHelper = shardHelper;
        this.handlers = handlers;
        this.moderationLogService = moderationLogService;
        this.propertiesSupport = propertiesSupport;
    }

    @Override
    protected void initializeBeforeReader() {
        monitoring = new ModerationReadMonitoring(getMetricRegistry());
    }

    @Override
    protected LogbrokerBatchReaderImpl<AbstractModerationResponse> getLogbrokerBatchReader(Supplier<SyncConsumer> syncConsumerSupplier) {
        return new ModerationResponseLogbrokerReader(syncConsumerSupplier, moderationParameters, getMetricRegistry(),
                new ModerationResponseSupportFacade(monitoring, typeSupports), propertiesSupport);
    }

    @Override
    protected BaseModerationResponseProcessor<AbstractModerationResponse> getEventsProcessor() {
        return new ModerationResponseProcessor(shardHelper, processorFilter, monitoring, handlers,
                moderationLogService);
    }
}
