package ru.yandex.direct.jobs.moderation;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderCloseException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl.LogbrokerBatchReaderImpl;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.AbstractLogbrokerWriterImpl;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriterRetryConfig;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.env.EnvironmentCondition;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.env.EnvironmentTypeProvider;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.env.TestingOnly;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.jobs.moderation.config.DestConfig;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.config.ResponseRoutingLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.config.ResponseRoutingParameters;
import ru.yandex.direct.jobs.moderation.config.TopicWithGroup;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.jobs.moderation.reader.ModerationResponseLogbrokerReader;
import ru.yandex.direct.jobs.moderation.reader.support.ModerationResponseConverter;
import ru.yandex.direct.jobs.moderation.reader.support.ModerationResponseSupportFacade;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.HourglassDaemon;
import ru.yandex.direct.scheduler.support.DirectParameterizedJob;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.json.JSonSplitter;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.jobs.configuration.ModerationConfiguration.MODERATION_ROUTER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

@SuppressWarnings("rawtypes")
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10),
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 20),
        tags = {DIRECT_PRIORITY_0},
        needCheck = TestingOnly.class
)
@HourglassDaemon(runPeriod = 0)
@Hourglass(periodInSeconds = 0, needSchedule = ReceiveModerationResponseRouterJob.ScheduleCondition.class)
@ParameterizedBy(parametersSource = ResponseRoutingLogbrokerConsumerPropertiesHolder.class)
@ParametersAreNonnullByDefault
public class ReceiveModerationResponseRouterJob extends DirectParameterizedJob<TopicWithGroup> {
    private static final Logger logger = LoggerFactory.getLogger(ReceiveModerationResponseRouterJob.class);

    // Сколько вердиктов можно записать в 1 сообщение
    private static final int MAX_VERDICTS_IN_MESSAGE = 100;
    // В какую группу партиций будем записывать вердикты без campaignId в meta (по идее таких не должно быть, но вдруг)
    public static final int DEFAULT_GROUP = 1;
    // Сколько времени ждать после ошибки в джобе
    public static final long SLEEP_ON_ERROR_MILLIS = Duration.ofMinutes(5).toMillis();
    // В этой джобе пачки сообщений берём поменьше, чтобы в случае ошибок не заспамить целевые топики
    public static final int ROWS_LIMIT = 5_000;
    // Таймаут записи в целевые топики
    public static final long WRITE_TIMEOUT_SECONDS = 60L;

    private final ResponseModerationParameters moderationParameters;
    private final ResponseRoutingLogbrokerConsumerPropertiesHolder consumerPropertiesHolder;
    private final ReceiveModerationService receiveModerationService;
    private final ShardHelper shardHelper;
    private final PpcPropertiesSupport propertiesSupport;
    private final List<ModerationResponseConverter> typeSupports;
    private final ResponseRoutingParameters routingParameters;
    private final LogbrokerClientFactoryFacade logbrokerClientFactory;

    private boolean isInitialized = false;

    private Labels solomonRegistryLabels;
    private MetricRegistry metricRegistry;
    private ModerationReadMonitoring monitoring;

    private RetryingLogbrokerBatchReader<AbstractModerationResponse> logbrokerReader;
    private Map<TopicWithGroup, ModerationResponseWriter> logbrokerWriters;

    /**
     * Условие на тип окружения, в котором задача должна работать.
     * Логика полностью повторяет условия для ReceiveModerationResponseJob
     */
    @Component
    public static class ScheduleCondition extends EnvironmentCondition {
        @Autowired
        public ScheduleCondition(EnvironmentTypeProvider environmentTypeProvider) {
            super(environmentTypeProvider, EnvironmentType.PRODUCTION, EnvironmentType.TESTING,
                    EnvironmentType.DEVTEST, EnvironmentType.DEVELOPMENT);
        }
    }

    @Autowired
    public ReceiveModerationResponseRouterJob(
            ResponseModerationParameters moderationParameters,
            ResponseRoutingLogbrokerConsumerPropertiesHolder consumerPropertiesHolder,
            ReceiveModerationService receiveModerationService,
            List<ModerationResponseConverter> typeSupports,
            ShardHelper shardHelper,
            PpcPropertiesSupport propertiesSupport,
            ResponseRoutingParameters routingParameters,
            @Qualifier(MODERATION_ROUTER_LOGBROKER_CLIENT_FACTORY_BEAN_NAME)
                    LogbrokerClientFactoryFacade logbrokerClientFactory) {
        this.moderationParameters = moderationParameters;
        this.consumerPropertiesHolder = consumerPropertiesHolder;
        this.receiveModerationService = receiveModerationService;
        this.typeSupports = typeSupports;
        this.shardHelper = shardHelper;
        this.propertiesSupport = propertiesSupport;
        this.routingParameters = routingParameters;
        this.logbrokerClientFactory = logbrokerClientFactory;
    }

    private void initializeInternal() {
        TopicWithGroup readTopic = consumerPropertiesHolder.convertStringToParam(getParam());
        LogbrokerConsumerProperties consumerProperties = consumerPropertiesHolder.getPropertiesByParam(readTopic);

        solomonRegistryLabels = Labels.of("jobs", "moderation_router", "topic", consumerProperties.getReadTopic());
        metricRegistry = SOLOMON_REGISTRY.subRegistry(solomonRegistryLabels);

        var syncConsumerSupplier = receiveModerationService.createConsumerSupplier(consumerProperties);

        monitoring = new ModerationReadMonitoring(getMetricRegistry());

        logbrokerReader = new RetryingLogbrokerBatchReader<>(
                () -> getLogbrokerBatchReader(syncConsumerSupplier),
                consumerProperties.getRetries());

        // Для каждой группы партиций каждого целевого топика создаём writer
        logbrokerWriters = createLogbrokerWriters();
    }

    Map<TopicWithGroup, ModerationResponseWriter> createLogbrokerWriters() {
        List<DestConfig> destConfigs = new ArrayList<>();
        destConfigs.add(routingParameters.getDefaultWriteTopic());
        destConfigs.addAll(routingParameters.getRoutesMap().values());

        // Для каждого целевого топика вычисляем максимальное количество используемых групп,
        // чтобы на все используемые группы завести writer'ы
        Map<String, Integer> topicsMaxGroups = EntryStream.of(StreamEx.of(destConfigs).groupingBy(DestConfig::getTopic))
                .mapValues(topics -> topics.stream().map(DestConfig::getGroupsCnt).max(Integer::compareTo).get())
                .toMap();

        return EntryStream.of(topicsMaxGroups)
                .flatMapValues(groupsCnt -> IntStream.rangeClosed(1, groupsCnt).boxed())
                .mapToKey(TopicWithGroup::new)
                .mapToValue((pair, __) -> createWriter(pair.getTopic(), pair.getGroup()))
                .toMap();
    }

    private ModerationResponseWriter createWriter(String writeTopic, int writeGroup) {
        // Разные инстансы джобы должны иметь разные sourceId
        // (иначе одновременно писать в одни и те же целевые топики не получится).
        // При этом для каждой партиции нужен свой sourceId,
        // т.к. они "прилипают" к партициям и не могут быть использованы для других партиций
        String sourceId = String.format("router-%s-%d", Objects.requireNonNull(getParam()), writeGroup);

        logger.info("Creating writer for topic {}:{} from sourceId={}", writeTopic, writeGroup, sourceId);

        var producerProperties = routingParameters.getProducerProperties();
        LogbrokerProducerProperties logbrokerProducerProperties = LogbrokerProducerPropertiesImpl.newBuilder()
                .setHost(producerProperties.getLogbrokerHost())
                .setCompressionCodec(producerProperties.getCompressionCodec())
                .setRetries(producerProperties.getRetries())
                .setTimeoutSec(producerProperties.getTimeoutSec())
                .setWriteTopic(writeTopic)
                .setGroup(writeGroup)
                .build();

        Supplier<CompletableFuture<AsyncProducer>> producerSupplier = logbrokerClientFactory.createProducerSupplier(
                logbrokerProducerProperties, sourceId);

        return new ModerationResponseWriter(producerSupplier,
                Duration.ofSeconds(logbrokerProducerProperties.getTimeoutSec()),
                logbrokerProducerProperties.getRetries());
    }

    public static class ModerationResponseWriter extends AbstractLogbrokerWriterImpl<List<AbstractModerationResponse>> {
        public ModerationResponseWriter(Supplier<CompletableFuture<AsyncProducer>> logbrokerProducerSupplier,
                                        Duration logbrokerTimeout,
                                        int retryCount) {
            super(logbrokerProducerSupplier, logbrokerTimeout, new LogbrokerWriterRetryConfig(retryCount));
        }

        @Override
        protected LogbrokerWriteRequest makeRequest(List<AbstractModerationResponse> batch) {
            String message = JSonSplitter.combineJsons(
                    batch.stream().map(AbstractModerationResponse::getOriginalJson).collect(toList()));
            return new LogbrokerWriteRequest(message.getBytes(StandardCharsets.UTF_8));
        }
    }

    private LogbrokerBatchReaderImpl<AbstractModerationResponse> getLogbrokerBatchReader(Supplier<SyncConsumer> syncConsumerSupplier) {
        return new ModerationResponseLogbrokerReader(syncConsumerSupplier, moderationParameters, getMetricRegistry(),
                new ModerationResponseSupportFacade(monitoring, typeSupports), propertiesSupport) {
            @Override
            protected Integer getRowsLimit() {
                return ROWS_LIMIT;
            }
        };
    }

    @Override
    public void execute() {
        if (!isInitialized) {
            initializeInternal();
            isInitialized = true;
        }
        //
        try {
            logbrokerReader.fetchEvents(getEventsProcessor());
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        }
    }

    DestConfig getDestinationTopic(AbstractModerationResponse object) {
        ModerationObjectType type = object.getType();
        return type != null
                ? routingParameters.getRoutesMap().getOrDefault(type, routingParameters.getDefaultWriteTopic())
                : routingParameters.getDefaultWriteTopic();
    }

    Map<Integer, List<AbstractModerationResponse>> groupByWriteGroup(List<AbstractModerationResponse> responses,
                                                                     DestConfig destConfig) {
        // Если вдруг найдутся объекты, у которых нет clientId, то их мы вынуждены будем положить
        // в первую группу партиций (т.к. не знаем, к какому шарду они относятся)
        // А вообще таких объектов быть не должно
        var withAndWithoutClientId =
                StreamEx.of(responses).partitioningBy(r -> r.getMeta().getClientId() != 0L);
        List<AbstractModerationResponse> withoutClientId = withAndWithoutClientId.getOrDefault(false, emptyList());
        List<AbstractModerationResponse> withClientId = withAndWithoutClientId.getOrDefault(true, emptyList());

        if (!withoutClientId.isEmpty()) {
            String types = withoutClientId.stream()
                    .map(AbstractModerationResponse::getType)
                    .distinct()
                    .map(t -> t != null ? t.getValue() : "null")
                    .collect(Collectors.joining(","));
            logger.warn("Found {} verdicts with clientId=0 (types: {})", withoutClientId.size(), types);
        }

        // Распределяем вердикты по группам партиций
        Map<Integer, List<AbstractModerationResponse>> writeGroupDataMap =
                StreamEx.of(IntStream.rangeClosed(1, destConfig.getGroupsCnt()).boxed())
                        .toMap(Function.identity(), group -> new ArrayList<>());
        groupByShard(withClientId).forEach((shard, shardResponses) -> {
            int writeGroup = (shard - 1) % destConfig.getGroupsCnt() + 1;
            writeGroupDataMap.get(writeGroup).addAll(shardResponses);
        });
        if (!withoutClientId.isEmpty()) {
            writeGroupDataMap.get(DEFAULT_GROUP).addAll(withoutClientId);
        }

        return writeGroupDataMap;
    }

    Map<Integer, List<AbstractModerationResponse>> groupByShard(List<AbstractModerationResponse> responses) {
        var groupedByShard = shardHelper.groupByShard(responses, ShardKey.CLIENT_ID, r -> r.getMeta().getClientId());
        return groupedByShard.getShardedDataMap();
    }

    private void routeResponses(List<AbstractModerationResponse> allResponses) {
        logger.info("Processing {} responses", allResponses.size());
        var groupedByTopic = StreamEx.of(allResponses).groupingBy(this::getDestinationTopic);

        List<CompletableFuture<Integer>> writingFutures = new ArrayList<>();
        AtomicInteger totalWritten = new AtomicInteger(0);
        for (var topicResponsesPair : groupedByTopic.entrySet()) {
            var destTopic = topicResponsesPair.getKey();
            var responses = topicResponsesPair.getValue();

            // Группируем вердикты по группам партиций
            var writeGroupDataMap = groupByWriteGroup(responses, destTopic);

            // Начинаем запись в группы партиций
            var futures = EntryStream.of(writeGroupDataMap)
                    .filterKeyValue((writeGroup, shardResponses) -> !shardResponses.isEmpty())
                    .mapKeyValue((writeGroup, shardResponses) -> {
                        String topic = destTopic.getTopic();
                        logger.info("Writing {} responses to {}:{}", shardResponses.size(), topic, writeGroup);
                        var writer = logbrokerWriters.get(new TopicWithGroup(topic, writeGroup));
                        totalWritten.addAndGet(shardResponses.size());
                        return writer.write(Lists.partition(shardResponses, MAX_VERDICTS_IN_MESSAGE));
                    })
                    .toList();
            writingFutures.addAll(futures);
        }

        // Ждём завершения всех операций записи
        try {
            logger.info("Waiting for {} responses will be written", totalWritten.get());
            CompletableFuture.allOf(writingFutures.toArray(new CompletableFuture[0]))
                    .get(WRITE_TIMEOUT_SECONDS, TimeUnit.SECONDS);
        } catch (RuntimeException | TimeoutException | ExecutionException e) {
            throw new RuntimeException("Writing failed", e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException("Writing failed", e);
        }
    }

    private BaseModerationResponseProcessor<AbstractModerationResponse> getEventsProcessor() {
        return new BaseModerationResponseProcessor<>(new ModerationResponseProcessorFilter(response -> true)) {
            @Override
            protected void processResponses(List<AbstractModerationResponse> responses) {
                try {
                    routeResponses(responses);
                } catch (RuntimeException e) {
                    // Если случилась ошибка, то подождём некоторое время, чтобы не заддосить целевые топики,
                    // в которые запись, возможно, прошла успешно
                    logger.error("Can't route responses, sleeping before next try", e);
                    try {
                        Thread.sleep(SLEEP_ON_ERROR_MILLIS);
                        // При ошибке нужно завершать итерацию джобы с ошибкой, чтобы jobs
                        // при следующем выполнении джобы создал и инициализировал новый экземпляр
                        // (в противном случае будут проблемы с состоянием reader'ов и writer'ов:
                        // reader'ы не будут повторно читать уже прочитанные сообщения, и мы можем просыпать на пол то,
                        // что не удалось обработать; а writer'ы могут просто падать с IllegalStateException
                        // из-за того, что пришли к невалидному внутреннему состоянию -- с этим нужно отдельно
                        // разбираться)
                        throw e;
                    } catch (InterruptedException ex) {
                        logger.error("Waiting interrupted", ex);
                        Thread.currentThread().interrupt();
                        throw new InterruptedRuntimeException(ex);
                    }
                }
            }
        };
    }

    /**
     * Возвращает {@link MetricRegistry} на конкретный топик логброкера
     */
    protected MetricRegistry getMetricRegistry() {
        checkState(metricRegistry != null, "metricRegistry not initialized");
        return metricRegistry;
    }

    @Override
    public void finish() {
        if (solomonRegistryLabels != null) {
            SOLOMON_REGISTRY.removeSubRegistry(solomonRegistryLabels);
        }
        if (logbrokerReader != null) {
            try {
                this.logbrokerReader.close();
            } catch (LogbrokerReaderCloseException ex) {
                logger.error("Error while closing logbrokerReader", ex);
            }
        }
        if (logbrokerWriters != null) {
            logbrokerWriters.values().forEach(logbrokerWriter -> {
                try {
                    logbrokerWriter.close();
                } catch (RuntimeException e) {
                    logger.error("Error closing logbrokerWriter", e);
                }
            });
        }
    }
}
