package ru.yandex.direct.jobs.moderation;

import java.util.List;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl.LogbrokerBatchReaderImpl;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.container.ModerationLogEntry;
import ru.yandex.direct.common.log.service.ModerationLogService;
import ru.yandex.direct.core.entity.moderation.model.ModerationUnparsed;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.jobs.moderation.config.UnparsedLogbrokerConsumerPropertiesHolder;
import ru.yandex.direct.jobs.moderation.processor.ModerationResponseProcessorFilter;
import ru.yandex.direct.jobs.moderation.reader.ModerationUnparsedLogbrokerReader;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.model.CheckTag;
import ru.yandex.direct.scheduler.Hourglass;
import ru.yandex.direct.scheduler.support.ParameterizedBy;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

/**
 * Джоба читает топик модерации, в которой присылаются заявки, которые не смогли быть распаршены.
 * Если в этой очереди есть записи, то нужно обратить внимание и исправить, т.к. возник рассинхрон формата у Директа
 * и Модерации и такие заявки не обрабатываются
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 30 + 1),
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION, CheckTag.DIRECT_PRODUCT_TEAM},
        needCheck = ProductionOnly.class
)
@Hourglass(periodInSeconds = 10 * 60, needSchedule = ProductionOnly.class)
@ParameterizedBy(parametersSource = UnparsedLogbrokerConsumerPropertiesHolder.class)
@ParametersAreNonnullByDefault
public class ReceiveModerationUnparsedResponseJob extends BaseReceiveModerationJob<ModerationUnparsed> {

    private final ModerationLogService moderationLogService;
    private final PpcPropertiesSupport propertiesSupport;

    @Autowired
    public ReceiveModerationUnparsedResponseJob(ResponseModerationParameters moderationParameters,
                                                ModerationResponseProcessorFilter processorFilter,
                                                UnparsedLogbrokerConsumerPropertiesHolder logbrokerConsumerProperties,
                                                ReceiveModerationService receiveModerationService,
                                                ModerationLogService moderationLogService,
                                                PpcPropertiesSupport propertiesSupport) {
        super(moderationParameters, processorFilter, logbrokerConsumerProperties, receiveModerationService);
        this.moderationLogService = moderationLogService;
        this.propertiesSupport = propertiesSupport;
    }

    @Override
    protected LogbrokerBatchReaderImpl<ModerationUnparsed> getLogbrokerBatchReader(Supplier<SyncConsumer> syncConsumerSupplier) {
        return new ModerationUnparsedLogbrokerReader(
                syncConsumerSupplier, moderationParameters, getMetricRegistry(), propertiesSupport);
    }

    @Override
    protected BaseModerationResponseProcessor<ModerationUnparsed> getEventsProcessor() {
        return new Processor(processorFilter, getMetricRegistry());
    }

    private class Processor extends BaseModerationResponseProcessor<ModerationUnparsed> {
        private static final String UNPARSED_RESPONSE = "unparsed_response";

        private final Counter unparsedResponseCount;

        Processor(ModerationResponseProcessorFilter filter, MetricRegistry metricRegistry) {
            super(filter);

            this.unparsedResponseCount = metricRegistry.counter(UNPARSED_RESPONSE);
        }

        @Override
        protected void processResponses(List<ModerationUnparsed> responses) {
            if (responses.isEmpty()) {
                setJugglerStatus(JugglerStatus.OK, "No new unparsed responses");
                return;
            }
            responses.forEach(error -> moderationLogService.logEvent(new ModerationLogEntry<>(0,
                    0, 0, ModerationLogEntry.Action.RESPONSE, ModerationLogEntry.Source.UNPARSED,
                    false, error)));
            unparsedResponseCount.add(responses.size());
            setJugglerStatus(JugglerStatus.CRIT, "moderation return unparsed response");
        }
    }
}
