package ru.yandex.direct.jobs.moderation.config;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;

import ru.yandex.direct.ess.common.logbroker.LogbrokerConsumerProperties;
import ru.yandex.direct.scheduler.support.ParametersSource;

import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;

public abstract class LogbrokerConsumerPropertiesHolder implements ParametersSource<TopicWithGroup> {
    private static final String PARAM_DELIMITER = "__";

    private final String host;
    private final String consumerName;
    private final int logbrokerDataTimeoutSec;
    private final int initTimeoutSec;
    private final int retries;
    private final Long readDataAfterTimestampMs;
    private final Map<String, List<Integer>> readTopicsGroups;

    public LogbrokerConsumerPropertiesHolder(String host, String readTopic, String consumerName,
                                             int logbrokerDataTimeoutSec, int initTimeoutSec, int retries,
                                             List<Integer> groups, Long readDataAfterTimestampMs) {
        this(host, consumerName, logbrokerDataTimeoutSec, initTimeoutSec, retries, readDataAfterTimestampMs,
                singletonMap(readTopic, groups));
    }

    public LogbrokerConsumerPropertiesHolder(String host,
                                             String consumerName,
                                             int logbrokerDataTimeoutSec,
                                             int initTimeoutSec,
                                             int retries,
                                             Long readDataAfterTimestampMs,
                                             Map<String, List<Integer>> readTopicsGroups) {
        Set<String> topics = readTopicsGroups.keySet();
        if (topics.stream().map(LogbrokerConsumerPropertiesHolder::getTopicAlias).count() < topics.size()) {
            throw new IllegalStateException("All topics should have unique aliases");
        }
        this.host = host;
        this.consumerName = consumerName;
        this.logbrokerDataTimeoutSec = logbrokerDataTimeoutSec;
        this.initTimeoutSec = initTimeoutSec;
        this.retries = retries;
        this.readDataAfterTimestampMs = readDataAfterTimestampMs;
        this.readTopicsGroups = readTopicsGroups;
    }

    public LogbrokerConsumerProperties getPropertiesByParam(TopicWithGroup param) {
        return getConsumerProperties(host, consumerName, logbrokerDataTimeoutSec,
                initTimeoutSec, retries, param.getGroup(), readDataAfterTimestampMs, param.getTopic());
    }

    private ModerationLogbrokerConsumerProperties getConsumerProperties(String logbrokerHost, String consumer,
                                                                        int logbrokerDataTimeoutSec,
                                                                        int logbrokerInitTimeoutSec,
                                                                        int retries,
                                                                        int group,
                                                                        Long readDataAfterTimestampMs,
                                                                        String logbrokerTopic) {
        return new ModerationLogbrokerConsumerProperties.Builder()
                .setHost(logbrokerHost)
                .setReadDataTimeoutSec(logbrokerDataTimeoutSec)
                .setInitTimeoutSec(logbrokerInitTimeoutSec)
                .setRetries(retries)
                .setGroups(singletonList(group))
                .setReadTopic(logbrokerTopic)
                .setConsumerName(consumer)
                .setReadDataAfterTimestampMs(readDataAfterTimestampMs)
                .build();
    }

    @Override
    public List<TopicWithGroup> getAllParamValues() {
        return EntryStream.of(readTopicsGroups)
                .flatMapValues(Collection::stream)
                .mapKeyValue(TopicWithGroup::new)
                .collect(Collectors.toList());
    }

    @Override
    public String convertParamToString(TopicWithGroup paramValue) {
        return String.format("%s%s%d", getTopicAlias(paramValue.getTopic()), PARAM_DELIMITER, paramValue.getGroup());
    }

    @Override
    public TopicWithGroup convertStringToParam(String string) {
        String[] parts = string.split(PARAM_DELIMITER);
        String alias = parts[0];
        String topic = readTopicsGroups.keySet().stream().filter(t -> getTopicAlias(t).equals(alias)).findFirst().get();
        int group = Integer.parseInt(parts[1]);
        return new TopicWithGroup(topic, group);
    }

    /**
     * Алиас топика используется в качестве ключа, которым параметризуются джобы приёма вердиктов
     * Полное название топика нельзя использовать в качестве параметра, т.к. слеш недопустим в названии juggler-события,
     * генерируемого для инстанса параметризованной джобы
     * <p>
     * В качестве алиаса используем название топика после последнего слеша
     */
    static String getTopicAlias(String topic) {
        int i = topic.lastIndexOf("/");
        if (i == -1) {
            return topic;
        }
        return topic.substring(i + 1);
    }
}
