package ru.yandex.direct.jobs.moderation.debug;

import java.util.Collection;
import java.util.List;

import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;

import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.jobs.configuration.DebugReceiveModerationJobRunnerConfiguration;
import ru.yandex.direct.jobs.moderation.BaseReceiveModerationJob;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.scheduler.hourglass.TaskParametersMap;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;

/**
 * Класс, позволяющий запустить чтение вердиктов модерации.
 * Требуется для запуска при отладке модерации.
 * <p>
 * <p>
 * Пример:
 * Program arguments: --cid 123 --pid 1234 --bid 12345
 */
public class DebugReceiveModerationJobRunner {
    private static final Logger logger = LoggingInitializer.getLogger(DebugReceiveModerationJobRunner.class);

    public static void main(String[] args) {
        DebugReceiveModerationParams debugParams = new DebugReceiveModerationParams();
        DebugReceiveModerationJobRunnerConfiguration.initializeParameters(debugParams);
        LoggingInitializer
                .initialize(debugParams.loggingInitializerParams,
                        DebugReceiveModerationJobRunner.class.getCanonicalName());
        ParserWithHelp.parse(DebugReceiveModerationJobRunner.class.getCanonicalName(), args, debugParams);

        debugParams.validate();

        try (AnnotationConfigApplicationContext ctx =
                     new AnnotationConfigApplicationContext(DebugReceiveModerationJobRunnerConfiguration.class)) {
            runReceiveModeration(ctx, debugParams);
        }
    }

    private static void runReceiveModeration(AnnotationConfigApplicationContext ctx,
                                             DebugReceiveModerationParams debugParams) {
        Class<? extends BaseReceiveModerationJob> job = getModerationJob(ctx, debugParams.moderationClassNames.get(0));
        logger.info("Found job {}", job.getCanonicalName());

        BaseReceiveModerationJob receiveModerationJob = ctx.getBean(job);
        receiveModerationJob.initialize(TaskParametersMap.of("param", "modadvert-dev-sm-results-log__1"));

        int workingTimeSec = debugParams.workingTimeSec * 1000;
        long startTime = System.currentTimeMillis();
        try {
            while (!Thread.currentThread().isInterrupted() && System.currentTimeMillis() - startTime <= workingTimeSec) {

                receiveModerationJob.execute();
                Thread.sleep(debugParams.sleepTimeSec * 1000);

            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } finally {
            receiveModerationJob.finish();
        }
    }

    private static Class<? extends BaseReceiveModerationJob> getModerationJob(ApplicationContext applicationContext,
                                                                              String moderationJobClassName) {
        Collection<BaseReceiveModerationJob> moderationJobs =
                applicationContext.getBeansOfType(BaseReceiveModerationJob.class).values();

        List<BaseReceiveModerationJob> matchedModerationJobs = moderationJobs.stream()
                .filter(r ->
                        r.getClass().getCanonicalName().equals(moderationJobClassName)
                                || r.getClass().getCanonicalName().endsWith("." + moderationJobClassName)
                )
                .collect(toList());

        checkState(!matchedModerationJobs.isEmpty(), "Receive moderation job bean is not found: %s",
                moderationJobClassName);
        checkState(matchedModerationJobs.size() == 1, "More than one job bean found");

        return matchedModerationJobs.get(0).getClass();
    }
}
