package ru.yandex.direct.jobs.moderation.debug;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.beust.jcommander.IStringConverter;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;
import com.beust.jcommander.ParametersDelegate;
import com.beust.jcommander.converters.LongConverter;

import ru.yandex.direct.logging.LoggingInitializerParams;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToInstant;

@ParametersAreNonnullByDefault
public class DebugReceiveModerationParams {

    @ParametersDelegate
    public LoggingInitializerParams loggingInitializerParams = new LoggingInitializerParams();

    @Parameter(
            names = {"--job"},
            description = "Class name of receive moderation job to run"
    )
    public List<String> moderationClassNames = new ArrayList<>();

    @Parameter(
            names = {"--cid"},
            description = "",
            converter = LongConverter.class
    )
    public Set<Long> campaignIds = new HashSet<>();


    @Parameter(
            names = {"--pid"},
            description = "",
            converter = LongConverter.class
    )
    public Set<Long> adgroupIds = new HashSet<>();

    @Parameter(
            names = {"--bid"},
            description = "",
            converter = LongConverter.class
    )
    public Set<Long> bannerIds = new HashSet<>();

    @Parameter(
            names = {"-t", "--working-time"},
            description = "Application working time in sec. Default - 60 sec."
    )
    public int workingTimeSec = 60;

    @Parameter(
            names = {"--sleep-time"},
            description = "sleep time between iterations in sec. Default - 30 sec."
    )
    public int sleepTimeSec = 30;

    @Parameter(
            names = {"--ess-tag"},
            description = "moderation responses with this tag will be processed by runner. If not specified, " +
                    "all requests will be processed."
    )
    public String essTag = "";

    @Parameter(
            names = {"--read-data-before-minutes"},
            description = "read data only for moderation responses sent before this time. Default - 30 min."
    )
    private Integer readDataSentToModerationBeforeMinutes;

    @Parameter(
            names = {"--read-only-data-for-requests-sent-after", "--read-data-sent-after"},
            description = "read data only for moderation responses sent after this time, e.g. '2020-07-16T15:00:00'",
            converter = DateTimeToTimestampMsConverter.class
    )
    private Long readDataSentAfterTimestampMs;

    public Long getReadDataSentAfterTimestampMs() {
        return readDataSentAfterTimestampMs;
    }

    public void validate() {
        checkArgument(moderationClassNames.size() == 1, "Supported only one rule class");
        checkArgument(sleepTimeSec > 5, "sleepTimeSec must be more than 5 sec");

        if (readDataSentToModerationBeforeMinutes == null && readDataSentAfterTimestampMs == null) {
            readDataSentToModerationBeforeMinutes = 30;
        }

        if (readDataSentToModerationBeforeMinutes != null) {
            if (readDataSentAfterTimestampMs != null) {
                throw new ParameterException("Parameters 'read-data-before-minutes' and " +
                        "'read-only-data-for-requests-sent-after' are mutually exclusive and can't be set both");
            }

            readDataSentAfterTimestampMs =
                    System.currentTimeMillis() - readDataSentToModerationBeforeMinutes * 60 * 1000;
        }
    }

    private static class DateTimeToTimestampMsConverter implements IStringConverter<Long> {
        @Override
        public Long convert(String value) {
            if (value == null || value.isEmpty()) {
                return null;
            }

            var datetime = LocalDateTime.parse(value, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
            return moscowDateTimeToInstant(datetime).toEpochMilli();
        }
    }

}
