package ru.yandex.direct.jobs.moderation.processor;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.log.container.ModerationLogEntry;
import ru.yandex.direct.common.log.service.ModerationLogService;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.core.entity.moderation.model.AdGroupModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.CampaignModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.callout.CalloutModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.mobilecontenticon.MobileContentIconModerationMeta;
import ru.yandex.direct.core.entity.moderation.service.ModerationObjectType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jobs.moderation.BaseModerationResponseProcessor;
import ru.yandex.direct.jobs.moderation.ModerationReadMonitoring;
import ru.yandex.direct.jobs.moderation.processor.handlers.ModerationResponseHandler;

import static java.util.function.Function.identity;
import static ru.yandex.direct.jobs.moderation.ModerationUtils.subtractByIdentity;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * По типу {@link AbstractModerationResponse} выбирает обрабочика заявки
 * Вердикт с неизвестным типом пропускается и логируется
 */
public class ModerationResponseProcessor extends BaseModerationResponseProcessor<AbstractModerationResponse> {
    private static final Logger logger = LoggerFactory.getLogger(ModerationResponseProcessor.class);

    private final ShardHelper shardHelper;
    private final ModerationReadMonitoring readMonitoring;
    private final Map<ModerationObjectType, ModerationResponseHandler> handlers;
    private final ModerationLogService moderationLogService;

    public ModerationResponseProcessor(ShardHelper shardHelper,
                                       ModerationResponseProcessorFilter filter,
                                       ModerationReadMonitoring readMonitoring,
                                       List<ModerationResponseHandler> handlers,
                                       ModerationLogService moderationLogService) {
        super(filter);
        this.shardHelper = shardHelper;
        this.readMonitoring = readMonitoring;
        this.handlers = listToMap(handlers, ModerationResponseHandler::getType);
        this.moderationLogService = moderationLogService;
    }

    @Override
    protected void processResponses(List<AbstractModerationResponse> responses) {
        logger.info("Starting processing {} moderation responses", responses.size());
        StreamEx.of(responses)
                .mapToEntry(AbstractModerationResponse::getType, identity())
                .grouping()
                .forEach((type, groupedResponses) -> {
                    ModerationResponseHandler handler = handlers.get(type);
                    if (handler == null) {
                        logger.warn("Unknown handler for response types: {}", listToSet(groupedResponses,
                                AbstractModerationResponse::getType));
                        return;
                    }
                    processGroupedModerationResponses(handler, groupedResponses);
                });
    }

    private void processGroupedModerationResponses(ModerationResponseHandler<AbstractModerationResponse> handler,
                                                   List<AbstractModerationResponse> responseList) {
        if (responseList.isEmpty()) {
            return;
        }
        logger.info("Handling {} moderation responses of type '{}'", responseList.size(), handler.getType());

        List<Exception> exceptions = new ArrayList<>();
        groupByShard(responseList, handler.getClientIDExtractor())
                .forEach((shard, responses) -> {
                    int responsesSize = responses.size();
                    try {
                        List<AbstractModerationResponse> successfulResponses = handler.handleResponses(shard,
                                responses, readMonitoring);
                        logResponses(successfulResponses, true);

                        List<AbstractModerationResponse> unsuccessfulResponses =
                                subtractByIdentity(responses, successfulResponses);

                        logResponses(unsuccessfulResponses, false);
                    } catch (Exception e) {
                        logResponses(responses, false);
                        exceptions.add(e);
                    }

                    if (responsesSize != responses.size()) {
                        logger.error("responses list was modified in handler {} for shard {} " +
                                        "(initial size: {}, final size: {})",
                                handler.getType(), shard, responsesSize, responses.size());
                    }
                });

        if (!exceptions.isEmpty()) {
            RuntimeException aggregatedException = new RuntimeException("Errors while handling responses");
            exceptions.forEach(aggregatedException::addSuppressed);
            throw aggregatedException;
        }
    }

    private void logResponses(List<AbstractModerationResponse> responses, boolean isSuccessful) {
        responses.forEach(response -> moderationLogService.logEvent(createLogEntry(response, isSuccessful)));
    }

    private <T extends AbstractModerationResponse> ModerationLogEntry<T> createLogEntry(
            T response, boolean isSuccessful) {
        long campaignId = 0;
        long adGroupId = 0;
        long bannerId = 0;

        ModerationMeta meta = response.getMeta();

        if (meta instanceof CampaignModerationMeta) {
            campaignId = ((CampaignModerationMeta) meta).getCampaignId();
        }
        if (meta instanceof AdGroupModerationMeta) {
            adGroupId = ((AdGroupModerationMeta) meta).getAdGroupId();
        }
        if (meta instanceof BaseBannerModerationMeta) {
            bannerId = ((BaseBannerModerationMeta) meta).getBannerId();
        }
        if (meta instanceof MobileContentIconModerationMeta) {
            bannerId = ((MobileContentIconModerationMeta) meta).getMobileContentId();
        }
        if (meta instanceof CalloutModerationMeta) {
            bannerId = ((CalloutModerationMeta) meta).getCalloutId();
        }

        return new ModerationLogEntry<>(
                campaignId, adGroupId, bannerId,
                ModerationLogEntry.Action.RESPONSE, ModerationLogEntry.Source.RESPONSE,
                isSuccessful, response);
    }

    /**
     * Группирует завяки по шардам на основе функции {@code clientIDExtractor}
     */
    private <T> Map<Integer, List<T>> groupByShard(List<T> responses, Function<T, Long> clientIDExtractor) {
        if (clientIDExtractor == null) {
            //фиктивная группировка для обработчика неизвестных вердиктов
            return Map.of(0, responses);
        }
        Map<Long, Integer> shardsByClientIds = shardHelper.getShardsByClientIds(mapList(responses, clientIDExtractor));

        return responses.stream().filter(r -> {
            Long clientId = clientIDExtractor.apply(r);
            Integer shard = shardsByClientIds.get(clientId);
            if (shard == null) {
                logger.warn("Shard not found for a clientID {}. Response {}", clientId, r);
                return false;
            }

            return true;
        }).collect(
                Collectors.groupingBy(r -> shardsByClientIds.get(clientIDExtractor.apply(r)),
                        Collectors.mapping(identity(), Collectors.toList())));
    }
}
