package ru.yandex.direct.jobs.moderation.processor.handlers;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import com.google.common.collect.Iterables;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResultResponse;
import ru.yandex.direct.core.entity.moderation.model.BaseModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ModerationResult;
import ru.yandex.direct.core.entity.moderation.service.receiving.ModerationReceivingService;
import ru.yandex.direct.jobs.moderation.ModerationReadMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public abstract class BaseModerationHandler<T extends AbstractModerationResultResponse<? extends BaseModerationMeta, ?
        extends ModerationResult>> implements ModerationResponseHandler<T> {

    private static final int CHUNK_SIZE = 1000;

    private final ModerationReceivingService<T> moderationReceivingService;

    public BaseModerationHandler(ModerationReceivingService<T> moderationReceivingService) {
        this.moderationReceivingService = moderationReceivingService;
    }

    @Override
    public Function<T, Long> getClientIDExtractor() {
        return e -> e.getMeta().getClientId();
    }

    @Override
    public List<T> handleResponses(int shard, List<T> responses, ModerationReadMonitoring readMonitoring) {
        int unknownVerdictsCount = 0;
        List<T> successfulVerdicts = new ArrayList<>();

        readMonitoring.incProcessedResponses(getType(), responses.size());

        try (TraceProfile profile = Trace.current().profile(getType().getValue() + "_moderation_response",
                String.valueOf(shard))) {

            for (var chunk : Iterables.partition(responses, CHUNK_SIZE)) {
                Pair<Integer, List<T>> result = moderationReceivingService.processModerationResponses(shard, chunk);
                unknownVerdictsCount += result.getLeft();
                successfulVerdicts.addAll(result.getRight());
            }
        }

        readMonitoring.incErrorUnknownModerationVerdict(unknownVerdictsCount);
        return successfulVerdicts;
    }

}
