package ru.yandex.direct.jobs.moderation.reader;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.BiPredicate;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.reader.impl.LogbrokerBatchReaderImpl;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.moderation.model.AbstractModerationResponse;
import ru.yandex.direct.jobs.moderation.config.ResponseModerationParameters;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageBatch;
import ru.yandex.kikimr.persqueue.consumer.transport.message.inbound.data.MessageData;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.direct.common.db.PpcPropertyNames.ESS_MODERATION_LOGBROKER_READER_ROWS_LIMIT;
import static ru.yandex.direct.utils.json.JSonSplitter.splitJsons;

/**
 * Базовый класс обработки вердиктов модерации
 * заявки приходят в виде строки {type:"one"} \n {type:"tow"}
 */
@ParametersAreNonnullByDefault
public abstract class BaseModerationLogbrokerReader<T extends AbstractModerationResponse> extends LogbrokerBatchReaderImpl<T> {
    private static final Logger logger = LoggerFactory.getLogger(BaseModerationLogbrokerReader.class);

    private static final int DEFAULT_ROWS_LIMIT = 1_000;
    private final PpcProperty<Integer> rowsLimitProperty;
    private final Supplier<Integer> rowsLimitCache;

    BaseModerationLogbrokerReader(Supplier<SyncConsumer> logbrokerConsumerSupplier,
                                  ResponseModerationParameters parameters,
                                  MetricRegistry metricRegistry,
                                  PpcPropertiesSupport propertiesSupport) {
        super(logbrokerConsumerSupplier, parameters.isLogbrokerNoCommit(), metricRegistry);

        this.rowsLimitProperty = propertiesSupport.get(ESS_MODERATION_LOGBROKER_READER_ROWS_LIMIT);
        this.rowsLimitCache = Suppliers.memoizeWithExpiration(() -> {
            try {
                int rowsLimit = rowsLimitProperty.getOrDefault(DEFAULT_ROWS_LIMIT);
                logger.info("Using rows limit = {}", rowsLimit);
                return rowsLimit;
            } catch (RuntimeException e) {
                logger.error("Error getting rows limit property, fallback to default", e);
                return DEFAULT_ROWS_LIMIT;
            }
        }, 5, TimeUnit.MINUTES);
    }

    /**
     * Разделяем набор json-ов на составные части и парсим {@code parseResponseJson()}
     */
    @Override
    protected List<T> batchDeserialize(MessageBatch messageBatch) {
        List<T> data = new ArrayList<>();

        for (MessageData messageData : messageBatch.getMessageData()) {
            String responseStr = new String(messageData.getDecompressedData(), UTF_8);

            logger.debug("JSON - {}", responseStr);

            String[] responses = splitJsons(responseStr);

            if (responses.length > 100) {
                logger.info("one moderation message has more than {} verdicts", responses.length);
            }

            for (String json : responses) {
                JsonNode node = preParseWithoutException(json);
                T response = parseResponseJson(node);
                if (response == null) {
                    continue;
                }
                response.setOriginalJson(json);

                data.add(response);

                logger.debug("Object - {}", response);
            }
        }

        logger.info("batchDeserialize verdicts count {}", data.size());
        return data;
    }

    protected Integer getRowsLimit() {
        return rowsLimitCache.get();
    }

    @Override
    protected BiPredicate<Long, Duration> batchingThreshold() {
        return (rows, time) -> rows < getRowsLimit() && time.compareTo(iterationTime) < 0;
    }

    protected abstract T parseResponseJson(@Nullable JsonNode node);

    @Override
    protected int count(List<T> e) {
        return e.size();
    }

    private JsonNode preParseWithoutException(String json) {
        try {
            return JsonUtils.fromJson(json);
        } catch (Exception e) {
            logger.warn("Invalid json {}", json, e);
            return null;
        }
    }
}
